# Hospital Admin Auto-Creation & Data Isolation Guide

## ✅ Features Implemented

### 1. Auto-Creation of Hospital Admin Users

When Super Admin adds a new hospital, an admin user is **automatically created** with:

- **Email Format**: `{hospital_code}@hospital.local`
  - Example: If hospital code is "H001", email = `h001@hospital.local`
  - If no code provided, uses: `hospital{id}@hospital.local`
  
- **Default Password**: `admin123`
  - Should be changed on first login
  
- **Name**: `{Hospital Name} Administrator`
  - Example: "Apollo Hospital Administrator"
  
- **Role**: `admin`
- **Hospital ID**: Automatically linked to the new hospital

### 2. Data Isolation

Each hospital's data is completely separate:
- ✅ Hospital A admin sees only Hospital A's data
- ✅ Hospital B admin sees only Hospital B's data
- ✅ Super Admin can see all hospitals' data

## 🔧 Implementation Details

### Code Location
File: `superadmin/app/controllers/SuperAdminController.php`
Method: `storeHospital()`

### Auto-Creation Logic
```php
// After hospital is created
$adminModel = new Admin();
$hospitalCode = strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $data['code'] ?? ''));
$defaultEmail = $hospitalCode . '@hospital.local';
$defaultPassword = 'admin123';
$adminData = [
    'name' => $data['name'] . ' Administrator',
    'email' => $defaultEmail,
    'role' => 'admin',
    'hospital_id' => $id, // Links admin to hospital
    'password_hash' => password_hash($defaultPassword, PASSWORD_BCRYPT),
];
$adminId = $adminModel->create($adminData);
```

## 📋 How to Use

### Adding a New Hospital (Super Admin)

1. Login to Super Admin Panel: `http://localhost/hms_project/superadmin/login`
2. Go to: **Hospitals → Add Hospital**
3. Fill in hospital details (name, code, address, etc.)
4. Click "Save"
5. **Automatically creates admin user with credentials shown in success message**

### Finding Admin Credentials

**Method 1: Success Message**
- After adding hospital, success message shows:
  ```
  Hospital added successfully! Admin Login - Email: h001@hospital.local | Password: admin123
  ```

**Method 2: Hospital View Page**
- Go to: **Hospitals → View Hospital**
- Scroll to "Hospital Admins" section
- See all admins for that hospital with their emails

**Method 3: Run Script**
- Browser: `http://localhost/hms_project/create_missing_hospital_admins.php`
- Creates admins for hospitals that don't have one
- Shows all credentials in a table

## 🛠️ Utility Scripts

### 1. Create Missing Hospital Admins
**URL**: `http://localhost/hms_project/create_missing_hospital_admins.php`

**Purpose**: 
- Creates admin users for hospitals that don't have one
- Shows all login credentials
- Useful if auto-creation failed or was skipped

### 2. Verify Data Isolation
**URL**: `http://localhost/hms_project/verify_data_isolation.php`

**Purpose**:
- Shows all hospitals and their admins
- Displays data distribution by hospital (patients, doctors, etc.)
- Verifies that `hospital_id` columns exist in all tables
- Helps identify data isolation issues

## ✅ Data Isolation Checklist

### Tables with `hospital_id` Column:
- ✅ `patients` - Each patient belongs to one hospital
- ✅ `doctors` - Each doctor belongs to one hospital
- ✅ `departments` - Each department belongs to one hospital
- ✅ `appointments` - Each appointment belongs to one hospital
- ✅ `rooms` - Each room belongs to one hospital
- ✅ `beds` - Each bed belongs to one hospital
- ✅ `admissions` - Each admission belongs to one hospital
- ✅ `invoices` - Each invoice belongs to one hospital
- ✅ `medicines` - Each medicine belongs to one hospital
- ✅ `services` - Each service belongs to one hospital
- ✅ `charges` - Each charge belongs to one hospital
- ✅ `lab_tests` - Each lab test belongs to one hospital
- ✅ `hr_staff` - Each staff member belongs to one hospital

### Controllers Filtering by `hospital_id`:
- ✅ `PatientsController` - Only shows own hospital's patients
- ✅ `DoctorsController` - Only shows own hospital's doctors
- ✅ `AppointmentsController` - Only shows own hospital's appointments
- ✅ `AdmissionsController` - Only shows own hospital's admissions
- ✅ `DepartmentsController` - Only shows own hospital's departments
- ✅ `RoomsController` - Only shows own hospital's rooms
- ✅ `DashboardController` - Only shows own hospital's stats

## 🔒 Security Features

1. **Session-Based Isolation**
   - Each hospital admin's session stores their `hospital_id`
   - All queries automatically filter by `hospital_id`

2. **Ownership Verification**
   - Before update/delete, system verifies ownership
   - Cross-hospital access is blocked

3. **Super Admin Access**
   - Super admin has `hospital_id = NULL`
   - Can see all hospitals' data
   - Can manage all hospitals

## 📝 Default Credentials Format

When a hospital is added:

| Hospital Code | Email | Password |
|--------------|-------|----------|
| H001 | h001@hospital.local | admin123 |
| APOLLO | apollo@hospital.local | admin123 |
| (no code, ID=5) | hospital5@hospital.local | admin123 |

## ⚠️ Important Notes

1. **Change Password**: Users should change password after first login
2. **Email Uniqueness**: Each admin email must be unique
3. **Hospital Code**: Providing a hospital code helps generate better emails
4. **Data Migration**: Existing data should have `hospital_id` set (use `update_existing_data_hospital_id.php`)

## 🐛 Troubleshooting

### Admin User Not Created

1. Check error logs: `error_log` entries in PHP error log
2. Run verification script: `verify_data_isolation.php`
3. Manually create: Use `create_missing_hospital_admins.php`

### Data Showing from Other Hospitals

1. Verify `hospital_id` column exists in table
2. Check controller is using `getHospitalId()`
3. Verify model is filtering by `hospital_id`
4. Run `verify_data_isolation.php` to check data distribution

### Admin Cannot Login

1. Verify admin was created: Check `admins` table
2. Check email format: Should match pattern `*@hospital.local`
3. Verify password hash is correct
4. Check `hospital_id` is set correctly in `admins` table

## 📞 Support

If you encounter issues:
1. Run `verify_data_isolation.php` to diagnose
2. Check PHP error logs
3. Verify database structure matches `database.sql`
4. Ensure all `hospital_id` columns exist

