# Multi-Hospital Data Isolation Implementation Guide

## Overview
Sabhi hospital ka data alag-alag isolated hona chahiye. Ye document batata hai ki kaise data isolation implement karein.

## Database Schema ✅
Sabhi tables mein `hospital_id` column already hai:
- ✅ patients
- ✅ doctors  
- ✅ departments
- ✅ appointments
- ✅ rooms
- ✅ beds
- ✅ admissions
- ✅ invoices/billing
- ✅ medicines/pharmacy
- ✅ lab_tests
- ✅ services
- ✅ charges
- ✅ hr_staff

## Implementation Pattern

### 1. Model Methods Pattern

**List/Query Methods:**
```php
public function list(array $opts = []): array
{
    $hospitalId = $opts['hospital_id'] ?? null;
    
    $where = [];
    $params = [];
    
    // ALWAYS filter by hospital_id if provided
    if ($hospitalId !== null) {
        $where[] = 'table_name.hospital_id = :hospital_id';
        $params[':hospital_id'] = $hospitalId;
    }
    
    // ... other filters ...
}
```

**Find Methods:**
```php
public function find(int $id, ?int $hospitalId = null): ?array
{
    $sql = "SELECT * FROM table_name WHERE id = :id";
    if ($hospitalId !== null) {
        $sql .= " AND hospital_id = :hospital_id";
    }
    // ... execute with params ...
}
```

**Create Methods:**
```php
public function create(array $data): int
{
    $sql = "INSERT INTO table_name (hospital_id, ...) VALUES (:hospital_id, ...)";
    // ALWAYS include hospital_id in INSERT
}
```

**All Methods:**
```php
public function all(?int $hospitalId = null): array
{
    if ($hospitalId !== null) {
        $stmt = $this->db()->prepare("SELECT * FROM table_name WHERE hospital_id = :hospital_id");
        $stmt->execute([':hospital_id' => $hospitalId]);
    } else {
        $stmt = $this->db()->query("SELECT * FROM table_name");
    }
}
```

### 2. Controller Methods Pattern

**Index/List:**
```php
public function index(): void
{
    $hospitalId = $this->getHospitalId(); // Get from session
    
    $filters = ['q' => $q, 'limit' => $perPage, 'offset' => $offset];
    if ($hospitalId !== null) {
        $filters['hospital_id'] = $hospitalId; // ALWAYS add hospital_id
    }
    
    $items = $model->list($filters);
}
```

**Store/Create:**
```php
public function store(): void
{
    $hospitalId = $this->getHospitalId();
    
    $data = [
        'hospital_id' => $hospitalId, // ALWAYS set hospital_id
        // ... other data ...
    ];
    
    $id = $model->create($data);
}
```

**Show/View:**
```php
public function show(): void
{
    $hospitalId = $this->getHospitalId();
    
    $item = $model->find($id, $hospitalId); // Pass hospital_id
    if (!$item) {
        // Not found - could be wrong hospital
    }
}
```

**Edit/Update:**
```php
public function edit(): void
{
    $hospitalId = $this->getHospitalId();
    
    $item = $model->find($id, $hospitalId); // Verify ownership
    // ... show form ...
}

public function update(): void
{
    $hospitalId = $this->getHospitalId();
    
    // Verify ownership before update
    $item = $model->find($id, $hospitalId);
    if (!$item) {
        // Not found
    }
    
    $model->update($id, $data);
}
```

**Delete:**
```php
public function delete(): void
{
    $hospitalId = $this->getHospitalId();
    
    // Verify ownership before delete
    $item = $model->find($id, $hospitalId);
    if (!$item) {
        // Not found
    }
    
    // Delete with hospital_id check
    $stmt = $model->db()->prepare('DELETE FROM table_name WHERE id = :id AND hospital_id = :hospital_id');
    $stmt->execute([':id' => $id, ':hospital_id' => $hospitalId]);
}
```

## Status of Implementation

### ✅ Completed Models/Controllers:
1. **Patient** - ✅ Complete (all methods have hospital_id filtering)
2. **Doctor** - ✅ Complete (all methods have hospital_id filtering)
3. **Appointment** - ✅ Complete (all methods have hospital_id filtering)
4. **Admission** - ✅ Complete (all methods have hospital_id filtering)
5. **Bed** - ✅ Complete (all methods have hospital_id filtering)
6. **Dashboard** - ✅ Complete (filters by hospital_id)

### 🟡 Partially Completed:
1. **Department** - ✅ Model updated, ⚠️ Controller needs completion
   - ✅ list() - has hospital_id
   - ✅ count() - has hospital_id
   - ✅ find() - has hospital_id
   - ✅ create() - has hospital_id
   - ✅ all() - has hospital_id
   - ✅ Controller index() - updated
   - ✅ Controller store() - updated
   - ✅ Controller show() - updated
   - ✅ Controller edit() - updated
   - ✅ Controller delete() - updated

### ❌ Needs Update:
1. **Room** - ❌ Needs hospital_id filtering
2. **Service** - ❌ Needs hospital_id filtering
3. **Charge** - ❌ Needs hospital_id filtering
4. **Billing/Invoice** - ❌ Needs hospital_id filtering
5. **Pharmacy/Medicine** - ❌ Needs hospital_id filtering
6. **Lab Test** - ❌ Needs hospital_id filtering
7. **HR Staff** - ❌ Needs hospital_id filtering

## Next Steps

1. Update Room model and controller
2. Update Service model and controller
3. Update Charge model and controller
4. Update Billing/Invoice model and controller
5. Update Pharmacy/Medicine model and controller
6. Update Lab Test model and controller
7. Update HR Staff model and controller

## Testing Checklist

For each module, test:
- [ ] Hospital A admin can only see Hospital A data
- [ ] Hospital B admin can only see Hospital B data
- [ ] Creating new records assigns correct hospital_id
- [ ] Updating records only works for own hospital
- [ ] Deleting records only works for own hospital
- [ ] Cross-hospital access is blocked

## Important Notes

1. **Super Admin**: Super admin ka `hospital_id` NULL hota hai, so they can see all data
2. **Hospital Admin**: Hospital admin ka `hospital_id` set hota hai, so they only see their hospital's data
3. **Validation**: Always verify ownership before update/delete operations
4. **Error Messages**: Don't reveal that data exists in another hospital (security)

