<?php
/**
 * Add hospital_id columns to all tables that need it
 * This script will add hospital_id column to tables that don't have it
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<!DOCTYPE html><html><head><title>Add Hospital ID Columns</title>";
echo "<style>body{font-family:Arial;padding:20px;background:#f5f5f5;}pre{background:#fff;padding:15px;border-radius:5px;box-shadow:0 2px 5px rgba(0,0,0,0.1);}</style>";
echo "</head><body>";
echo "<h2>Add Hospital ID Columns to Tables</h2>";
echo "<pre>";

// Load database config
$dbConfigFile = __DIR__ . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'database.php';
if (!file_exists($dbConfigFile)) {
    die("✗ Database config file not found\n");
}

$dbConfig = require $dbConfigFile;

// Extract connection details
$host = '127.0.0.1';
$port = 3306;
$dbname = 'hms';
$user = $dbConfig['user'] ?? 'root';
$pass = $dbConfig['pass'] ?? '';

if (isset($dbConfig['dsn'])) {
    if (preg_match('/host=([^;]+)/', $dbConfig['dsn'], $matches)) {
        $host = $matches[1];
    }
    if (preg_match('/port=(\d+)/', $dbConfig['dsn'], $matches)) {
        $port = (int)$matches[1];
    }
    if (preg_match('/dbname=([^;]+)/', $dbConfig['dsn'], $matches)) {
        $dbname = $matches[1];
    }
}

try {
    // Connect to database
    echo "Connecting to database...\n";
    $pdo = new PDO(
        "mysql:host={$host};port={$port};dbname={$dbname};charset=utf8mb4",
        $user,
        $pass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    echo "✓ Connected to database '{$dbname}'\n\n";

    // Check if hospitals table exists
    echo "Step 1: Checking hospitals table...\n";
    $stmt = $pdo->query("SHOW TABLES LIKE 'hospitals'");
    if (!$stmt->fetch()) {
        echo "✗ Hospitals table does not exist. Please run setup_database_mvc.php first.\n";
        exit;
    }
    echo "✓ Hospitals table exists\n\n";

    // Tables that need hospital_id column
    $tables = [
        'patients' => ['after' => 'id', 'nullable' => true],
        'doctors' => ['after' => 'id', 'nullable' => true],
        'departments' => ['after' => 'id', 'nullable' => true],
        'appointments' => ['after' => 'id', 'nullable' => true],
        'rooms' => ['after' => 'id', 'nullable' => true],
        'beds' => ['after' => 'id', 'nullable' => true],
        'admissions' => ['after' => 'id', 'nullable' => true],
        'invoices' => ['after' => 'id', 'nullable' => true],
        'medicines' => ['after' => 'id', 'nullable' => true],
        'pharmacy_sales' => ['after' => 'id', 'nullable' => true],
        'lab_tests' => ['after' => 'id', 'nullable' => true],
        'lab_orders' => ['after' => 'id', 'nullable' => true],
        'services' => ['after' => 'id', 'nullable' => true],
        'charges' => ['after' => 'id', 'nullable' => true],
        'hr_staff' => ['after' => 'id', 'nullable' => true],
        'doctor_schedules' => ['after' => 'doctor_id', 'nullable' => true],
    ];

    $added = [];
    $skipped = [];
    $errors = [];

    echo "Step 2: Adding hospital_id columns...\n\n";

    foreach ($tables as $table => $config) {
        // Check if table exists
        $stmt = $pdo->query("SHOW TABLES LIKE '{$table}'");
        if (!$stmt->fetch()) {
            echo "  ⊙ {$table} - Table does not exist, skipping...\n";
            $skipped[] = $table;
            continue;
        }

        // Check if hospital_id column already exists
        $stmt = $pdo->query("SHOW COLUMNS FROM {$table} LIKE 'hospital_id'");
        if ($stmt->fetch()) {
            echo "  ⊙ {$table} - hospital_id column already exists\n";
            $skipped[] = $table;
            continue;
        }

        // Add hospital_id column
        try {
            $after = $config['after'] ?? 'id';
            $nullable = $config['nullable'] ?? true;
            $nullClause = $nullable ? 'NULL' : 'NOT NULL';
            
            $sql = "ALTER TABLE `{$table}` ADD COLUMN `hospital_id` INT UNSIGNED {$nullClause} AFTER `{$after}`";
            $pdo->exec($sql);
            
            // Add index
            try {
                $pdo->exec("ALTER TABLE `{$table}` ADD KEY `idx_{$table}_hospital` (`hospital_id`)");
            } catch (PDOException $e) {
                // Index might already exist or fail, continue
            }
            
            // Add foreign key if hospitals table exists
            try {
                $pdo->exec("ALTER TABLE `{$table}` ADD CONSTRAINT `fk_{$table}_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE SET NULL");
            } catch (PDOException $e) {
                // Foreign key might fail, continue
            }
            
            echo "  ✓ {$table} - Added hospital_id column\n";
            $added[] = $table;
        } catch (PDOException $e) {
            echo "  ✗ {$table} - Error: " . $e->getMessage() . "\n";
            $errors[] = $table;
        }
    }

    // Summary
    echo "\n";
    echo "========================================\n";
    echo "Summary\n";
    echo "========================================\n";
    echo "Columns added: " . count($added) . "\n";
    if (!empty($added)) {
        echo "\nAdded to tables:\n";
        foreach ($added as $table) {
            echo "  - {$table}\n";
        }
    }
    echo "\nSkipped: " . count($skipped) . " table(s)\n";
    if (!empty($errors)) {
        echo "\nErrors: " . count($errors) . " table(s)\n";
    }

    echo "\n";
    echo "========================================\n";
    echo "✓ Complete!\n";
    echo "========================================\n";
    echo "\n";
    echo "All tables now have hospital_id column.\n";
    echo "You can now run update_existing_data_hospital_id.php to set hospital_id = 1 for existing data.\n";
    
} catch (Exception $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Error!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
}

echo "</pre>";
echo "</body></html>";
?>

