<?php
declare(strict_types=1);

class AdmissionsController extends Controller
{
    public function index(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        $flashSuccess = $_SESSION['flash_success'] ?? null;
        if (isset($_SESSION['flash_success'])) { unset($_SESSION['flash_success']); }

        $q = trim((string)($_GET['q'] ?? ''));
        $doctor = isset($_GET['doctor']) ? (int)$_GET['doctor'] : null;
        $status = trim((string)($_GET['status'] ?? ''));
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $model = new Admission();
        $admissions = $model->list(['q' => $q, 'doctor_id' => $doctor, 'status' => $status, 'limit' => $perPage, 'offset' => $offset]);
        $total = $model->count(['q' => $q, 'doctor_id' => $doctor, 'status' => $status]);

        $this->view('admissions/list', [
            'title' => 'IPD Admissions',
            'admissions' => $admissions,
            'doctors' => $model->doctors(),
            'q' => $q,
            'doctor' => $doctor,
            'status' => $status,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
            'flashSuccess' => $flashSuccess,
        ]);
    }

    public function add(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        $formError = $_SESSION['form_error'] ?? null;
        $formErrors = $_SESSION['form_errors'] ?? [];
        $old = $_SESSION['old'] ?? null;
        if (isset($_SESSION['form_error'])) { unset($_SESSION['form_error']); }
        if (isset($_SESSION['form_errors'])) { unset($_SESSION['form_errors']); }
        if (isset($_SESSION['old'])) { unset($_SESSION['old']); }

        $patientId = isset($_GET['patient_id']) ? (int)$_GET['patient_id'] : 0;
        $patient = null;
        if ($patientId > 0) {
            $patientModel = new Patient();
            $patient = $patientModel->find($patientId);
        }

        $model = new Admission();
        $appointmentModel = new Appointment();
        $this->view('admissions/add', [
            'title' => 'IPD Admission',
            'patients' => $model->patients(),
            'doctors' => $appointmentModel->doctors(),
            'departments' => (new Department())->all(),
            'beds' => $model->availableBeds(),
            'selectedPatient' => $patient,
            'formError' => $formError,
            'formErrors' => $formErrors,
            'old' => $old,
        ]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $hospitalId = $this->getHospitalId();

        $data = [
            'hospital_id' => $hospitalId,
            'patient_id' => (int)($_POST['patient_id'] ?? 0),
            'doctor_id' => (int)($_POST['doctor_id'] ?? 0),
            'bed_id' => (int)($_POST['bed_id'] ?? 0),
            'diagnosis' => trim($_POST['diagnosis'] ?? ''),
            'admission_type' => trim($_POST['admission_type'] ?? 'Emergency'),
            'notes' => trim($_POST['notes'] ?? ''),
        ];

        $errors = $this->validateAdmission($data);

        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/admissions/add' . (!empty($data['patient_id']) ? '?patient_id=' . urlencode((string)$data['patient_id']) : ''));
            exit;
        }

        try {
            $data['doctor_id'] = $data['doctor_id'] ?: null;
            $data['bed_id'] = $data['bed_id'] ?: null;
            $model = new Admission();
            $id = $model->create($data);

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Patient admitted successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/admissions/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            error_log('IPD Admission error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not admit patient: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/admissions/add' . (!empty($data['patient_id']) ? '?patient_id=' . urlencode((string)$data['patient_id']) : ''));
            exit;
        }
    }

    public function show(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid admission id';
            return;
        }

        $hospitalId = $this->getHospitalId();
        $model = new Admission();
        $admission = $model->find($id, $hospitalId);

        if (!$admission) {
            http_response_code(404);
            echo 'Admission not found';
            return;
        }

        $this->view('admissions/view', [
            'title' => 'IPD Admission Details',
            'admission' => $admission,
        ]);
    }

    public function discharge(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) {
            http_response_code(422);
            echo 'Invalid id';
            return;
        }

        try {
            (new Admission())->discharge($id);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Patient discharged successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/admissions/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            error_log('Discharge error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not discharge patient: ' . $e->getMessage();
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/admissions/view?id=' . urlencode((string)$id));
            exit;
        }
    }

    protected function validateAdmission(array $data): array
    {
        $errors = [];

        if (empty($data['patient_id']) || $data['patient_id'] <= 0) {
            $errors['patient_id'] = 'Patient is required.';
        }

        if (!empty($data['bed_id']) && $data['bed_id'] <= 0) {
            $errors['bed_id'] = 'Invalid bed selection.';
        }

        return $errors;
    }
}
