<?php
declare(strict_types=1);

class AttendanceController extends Controller
{
    public function index(): void
    {
        $date = $_GET['date'] ?? date('Y-m-d');
        $month = $_GET['month'] ?? date('Y-m');
        $departmentId = isset($_GET['department_id']) && (int)$_GET['department_id'] > 0 ? (int)$_GET['department_id'] : null;

        $attendance = [];
        $staff = [];
        $departments = [];
        $monthlyReport = [];

        try {
            $attendanceModel = new Attendance();
            $attendance = $attendanceModel->getAttendanceByDate($date);
            $monthlyReport = $attendanceModel->getMonthlyReport($month, $departmentId);

            $staffModel = new Staff();
            $staff = $staffModel->all();
            $departments = $staffModel->getDepartments();
        } catch (Throwable $e) {
            $attendance = [];
            $staff = [];
            $departments = [];
            $monthlyReport = [];
        }

        $this->view('attendance/index', [
            'title' => 'Attendance Management',
            'date' => $date,
            'month' => $month,
            'department_id' => $departmentId,
            'attendance' => $attendance,
            'staff' => $staff,
            'departments' => $departments,
            'monthlyReport' => $monthlyReport,
        ]);
    }

    public function mark(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $staffId = isset($_POST['staff_id']) ? (int)$_POST['staff_id'] : 0;
        $date = trim($_POST['date'] ?? date('Y-m-d'));
        $checkIn = trim($_POST['check_in'] ?? '') ?: null;
        $checkOut = trim($_POST['check_out'] ?? '') ?: null;
        $status = trim($_POST['status'] ?? 'Present');
        $remarks = trim($_POST['remarks'] ?? '') ?: null;

        if ($staffId <= 0) {
            http_response_code(400);
            echo 'Invalid staff ID';
            return;
        }

        try {
            $model = new Attendance();
            $model->markAttendance($staffId, $date, $checkIn, $checkOut, $status, $remarks);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Attendance marked successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/attendance?date=' . urlencode($date));
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not mark attendance: ' . $e->getMessage();
        }
    }

    public function bulkMark(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $date = trim($_POST['date'] ?? date('Y-m-d'));
        $attendanceData = $_POST['attendance'] ?? [];

        if (empty($attendanceData)) {
            http_response_code(400);
            echo 'No attendance data provided';
            return;
        }

        try {
            $model = new Attendance();
            $count = 0;
            foreach ($attendanceData as $staffId => $data) {
                $staffId = (int)$staffId;
                if ($staffId <= 0) continue;

                $checkIn = !empty($data['check_in']) ? $data['check_in'] : null;
                $checkOut = !empty($data['check_out']) ? $data['check_out'] : null;
                $status = $data['status'] ?? 'Present';
                $remarks = !empty($data['remarks']) ? $data['remarks'] : null;

                $model->markAttendance($staffId, $date, $checkIn, $checkOut, $status, $remarks);
                $count++;
            }

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = "Attendance marked for {$count} staff member(s).";
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/attendance?date=' . urlencode($date));
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not mark attendance: ' . $e->getMessage();
        }
    }

    public function update(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid attendance ID';
            return;
        }

        $data = [
            'check_in' => trim($_POST['check_in'] ?? '') ?: null,
            'check_out' => trim($_POST['check_out'] ?? '') ?: null,
            'status' => trim($_POST['status'] ?? 'Present'),
            'remarks' => trim($_POST['remarks'] ?? '') ?: null,
        ];

        try {
            $model = new Attendance();
            $model->update($id, $data);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Attendance updated successfully.';
            $date = $_POST['date'] ?? date('Y-m-d');
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/attendance?date=' . urlencode($date));
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not update attendance: ' . $e->getMessage();
        }
    }
}






