<?php
declare(strict_types=1);

class BedsController extends Controller
{
    public function index(): void
    {
        $db = (new Bed())->db();
        
        // Get bed allocations with patient, room, and admission details
        $sql = "SELECT 
                    b.id AS bed_id,
                    b.ward,
                    b.number AS bed_number,
                    b.type AS bed_type,
                    b.is_available,
                    r.id AS room_id,
                    r.name AS room_name,
                    r.code AS room_code,
                    r.charge_per_day,
                    a.id AS admission_id,
                    a.admitted_at,
                    a.discharged_at,
                    a.diagnosis,
                    p.id AS patient_id,
                    CONCAT(p.first_name, ' ', p.last_name) AS patient_name,
                    CASE 
                        WHEN a.discharged_at IS NULL AND a.id IS NOT NULL THEN 'Occupied'
                        ELSE 'Available'
                    END AS status
                FROM beds b
                LEFT JOIN rooms r ON r.id = b.room_id
                LEFT JOIN admissions a ON a.bed_id = b.id AND a.discharged_at IS NULL
                LEFT JOIN patients p ON p.id = a.patient_id
                ORDER BY b.ward, b.number";
        
        $stmt = $db->query($sql);
        $allocations = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Format dates and calculate duration
        foreach ($allocations as &$allocation) {
            if (!empty($allocation['admitted_at'])) {
                $admitDate = new DateTime($allocation['admitted_at']);
                $allocation['admit_date_formatted'] = $admitDate->format('d-m-Y');
                $allocation['admit_time'] = $admitDate->format('H:i');
                
                // Calculate duration
                $now = new DateTime();
                $duration = $admitDate->diff($now);
                $allocation['duration_days'] = $duration->days;
                $allocation['duration_text'] = $duration->days . ' day(s)';
                if ($duration->days === 0) {
                    $allocation['duration_text'] = 'Today';
                } elseif ($duration->days === 1) {
                    $allocation['duration_text'] = '1 day';
                }
            } else {
                $allocation['admit_date_formatted'] = null;
                $allocation['duration_text'] = null;
            }
            
            // Format charges
            $allocation['charge_display'] = '₹' . number_format((float)($allocation['charge_per_day'] ?? 0), 2) . '/day';
        }
        unset($allocation);
        
        $this->view('beds/list', [
            'title' => 'Bed Allocation',
            'allocations' => $allocations,
            'base' => defined('BASE_URL') ? BASE_URL : '',
        ]);
    }

    public function add(): void
    {
        $this->view('beds/add', [
            'title' => 'Add Bed',
        ]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $data = [
            'ward' => trim($_POST['ward'] ?? ''),
            'number' => trim($_POST['number'] ?? ''),
            'type' => trim($_POST['type'] ?? ''),
        ];
        if ($data['ward'] === '' || $data['number'] === '') {
            http_response_code(422);
            echo 'Ward and Number are required.';
            return;
        }
        $id = (new Bed())->create($data);
        header('Location: /admin/beds');
        exit;
    }
}
