<?php
declare(strict_types=1);

class BillingController extends Controller
{
    public function index(): void
    {
        $model = new Invoice();
        $invoices = $model->all();
        $this->view('billing/list', [
            'title' => 'Billing',
            'invoices' => $invoices,
        ]);
    }

    public function add(): void
    {
        $model = new Invoice();
        $this->view('billing/add', [
            'title' => 'Create Invoice',
            'patients' => $model->patients(),
        ]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $patientId = (int)($_POST['patient_id'] ?? 0);
        $items = $_POST['items'] ?? [];
        if ($patientId <= 0 || empty($items)) {
            http_response_code(422);
            echo 'Patient and at least one item required.';
            return;
        }
        $normalized = [];
        foreach ($items as $it) {
            $normalized[] = [
                'description' => trim($it['description'] ?? ''),
                'qty' => (float)($it['qty'] ?? 0),
                'unit_price' => (float)($it['unit_price'] ?? 0),
            ];
        }
        $invoiceId = (new Invoice())->create($patientId, $normalized);
        header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/billing/view?id=' . urlencode((string)$invoiceId));
        exit;
    }

    public function show(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid invoice id';
            return;
        }
        $invoice = (new Invoice())->find($id);
        if (!$invoice) {
            http_response_code(404);
            echo 'Invoice not found';
            return;
        }
        $this->view('billing/view', [
            'title' => 'Invoice #' . $id,
            'invoice' => $invoice,
        ]);
    }

    public function addPayment(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $invoiceId = (int)($_POST['invoice_id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        $method = trim($_POST['method'] ?? '');
        if ($invoiceId <= 0 || $amount <= 0) {
            http_response_code(422);
            echo 'Valid invoice id and amount required.';
            return;
        }
        (new Invoice())->addPayment($invoiceId, $amount, $method ?: null);
        header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/billing/view?id=' . urlencode((string)$invoiceId));
        exit;
    }

    // Submodules
    public function payments(): void
    {
        $this->view('billing/payments', [
            'title' => 'Billing - Payments',
        ]);
    }

    public function refunds(): void
    {
        $this->view('billing/refunds', [
            'title' => 'Billing - Refunds',
        ]);
    }

    public function items(): void
    {
        $this->view('billing/items', [
            'title' => 'Billing - Items & Services',
        ]);
    }

    public function estimates(): void
    {
        $this->view('billing/estimates', [
            'title' => 'Billing - Estimates',
        ]);
    }
}
