<?php
declare(strict_types=1);

class ChargesController extends Controller
{
    public function index(): void
    {
        // list charges (search, filter, pagination)
        $q = trim((string)($_GET['q'] ?? ''));
        $status = isset($_GET['status']) && $_GET['status'] !== '' ? (int)$_GET['status'] : null;
        $chargeType = isset($_GET['charge_type']) && $_GET['charge_type'] !== '' ? (string)$_GET['charge_type'] : null;
        $departmentId = isset($_GET['department_id']) && $_GET['department_id'] !== '' ? (int)$_GET['department_id'] : null;
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $charges = [];
        $total = 0;
        $departments = [];
        $chargeTypes = [];
        try {
            $model = new Charge();
            $charges = $model->list([
                'q' => $q, 
                'status' => $status, 
                'charge_type' => $chargeType,
                'department_id' => $departmentId, 
                'limit' => $perPage, 
                'offset' => $offset
            ]);
            $total = $model->count([
                'q' => $q, 
                'status' => $status, 
                'charge_type' => $chargeType,
                'department_id' => $departmentId
            ]);
            
            $chargeTypes = $model->getChargeTypes();
            
            $deptModel = new Department();
            $deptList = $deptModel->list(['status' => 1, 'limit' => 1000]);
            $departments = $deptList;
        } catch (Throwable $e) {
            $charges = [];
            $total = 0;
            $departments = [];
            $chargeTypes = [];
        }

        $this->view('charges/list', [
            'title' => 'Charges Management',
            'charges' => $charges,
            'departments' => $departments,
            'chargeTypes' => $chargeTypes,
            'q' => $q,
            'status' => $status,
            'charge_type' => $chargeType,
            'department_id' => $departmentId,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
        ]);
    }

    public function create(): void
    {
        // show add form
        $departments = [];
        $chargeTypes = [];
        try {
            $model = new Charge();
            $chargeTypes = $model->getChargeTypes();
            
            $deptModel = new Department();
            $departments = $deptModel->list(['status' => 1, 'limit' => 1000]);
        } catch (Throwable $e) {
            $departments = [];
            $chargeTypes = [];
        }
        $this->view('charges/add', [
            'title' => 'Add Charge',
            'departments' => $departments,
            'chargeTypes' => $chargeTypes,
        ]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        if (session_status() !== PHP_SESSION_ACTIVE) { 
            session_start(); 
        }
        unset($_SESSION['form_error'], $_SESSION['old']);

        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? '') ?: null,
            'charge_type' => trim($_POST['charge_type'] ?? 'Other'),
            'department_id' => isset($_POST['department_id']) && (int)$_POST['department_id'] > 0 ? (int)$_POST['department_id'] : null,
            'category' => trim($_POST['category'] ?? '') ?: null,
            'description' => trim($_POST['description'] ?? '') ?: null,
            'amount' => isset($_POST['amount']) ? (float)$_POST['amount'] : 0.00,
            'tax_percentage' => isset($_POST['tax_percentage']) ? (float)$_POST['tax_percentage'] : 0.00,
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];

        // validate input
        $errors = $this->validateCharge($data);
        if (!empty($errors)) {
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/charges/add');
            exit;
        }

        try {
            $model = new Charge();
            $id = $model->create($data);
            $_SESSION['flash_success'] = 'Charge created successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/charges/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            $_SESSION['old'] = $data;
            $msg = 'Could not save charge.';
            if ($e instanceof PDOException) {
                $info = $e->errorInfo ?? null;
                if (is_array($info) && isset($info[1]) && intval($info[1]) === 1062) {
                    $msg = 'Charge code already exists.';
                }
            }
            $_SESSION['form_error'] = $msg;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/charges/add');
            exit;
        }
    }

    public function show(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid charge id'; 
            return; 
        }
        try {
            $model = new Charge();
            $charge = $model->find($id);
            if (!$charge) { 
                http_response_code(404); 
                echo 'Charge not found'; 
                return; 
            }
            $this->view('charges/view', [
                'title' => 'Charge Details',
                'charge' => $charge,
            ]);
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Error loading charge'; 
        }
    }

    public function edit(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid charge id'; 
            return; 
        }
        try {
            $model = new Charge();
            $charge = $model->find($id);
            if (!$charge) { 
                http_response_code(404); 
                echo 'Charge not found'; 
                return; 
            }
            $chargeTypes = $model->getChargeTypes();
            
            $deptModel = new Department();
            $departments = $deptModel->list(['status' => 1, 'limit' => 1000]);
            $this->view('charges/edit', [
                'title' => 'Edit Charge',
                'charge' => $charge,
                'departments' => $departments,
                'chargeTypes' => $chargeTypes,
            ]);
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Error loading charge for edit'; 
        }
    }

    public function update(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid charge id'; 
            return; 
        }
        
        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? '') ?: null,
            'charge_type' => trim($_POST['charge_type'] ?? 'Other'),
            'department_id' => isset($_POST['department_id']) && (int)$_POST['department_id'] > 0 ? (int)$_POST['department_id'] : null,
            'category' => trim($_POST['category'] ?? '') ?: null,
            'description' => trim($_POST['description'] ?? '') ?: null,
            'amount' => isset($_POST['amount']) ? (float)$_POST['amount'] : 0.00,
            'tax_percentage' => isset($_POST['tax_percentage']) ? (float)$_POST['tax_percentage'] : 0.00,
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];
        
        // validate input (exclude current id when checking duplicates)
        $errors = $this->validateCharge($data, $id);
        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/charges/edit?id=' . urlencode((string)$id));
            exit;
        }
        
        try {
            $model = new Charge();
            $result = $model->update($id, $data);
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Charge updated successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/charges');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500);
            error_log('Charge update error: ' . $e->getMessage());
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['old'] = $data;
            $_SESSION['form_error'] = 'Could not update charge: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/charges/edit?id=' . urlencode((string)$id));
            exit;
        }
    }

    public function delete(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid id'; 
            return; 
        }
        try {
            $model = new Charge();
            $stmt = $model->db()->prepare('DELETE FROM charges WHERE id = :id');
            $stmt->execute([':id' => $id]);
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/charges');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Could not delete charge.'; 
        }
    }

    public function deactivate(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid id'; 
            return; 
        }
        try {
            $model = new Charge();
            $stmt = $model->db()->prepare('UPDATE charges SET status = 0 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Charge deactivated.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/charges');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Could not deactivate charge.'; 
        }
    }

    public function restore(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid id'; 
            return; 
        }
        try {
            $model = new Charge();
            $stmt = $model->db()->prepare('UPDATE charges SET status = 1 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Charge restored.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/charges');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Could not restore charge.'; 
        }
    }

    /**
     * Validate charge input. Returns array of field => message.
     * If $excludeId is provided it will be ignored when checking duplicate name/code.
     */
    protected function validateCharge(array $data, ?int $excludeId = null): array
    {
        $errors = [];
        // name required
        if (trim((string)($data['name'] ?? '')) === '') {
            $errors['name'] = 'Name is required.';
        } elseif (mb_strlen($data['name']) > 200) {
            $errors['name'] = 'Name is too long (max 200 chars).';
        }

        // code optional but max length
        if (isset($data['code']) && $data['code'] !== '' && mb_strlen($data['code']) > 50) {
            $errors['code'] = 'Code is too long (max 50 chars).';
        }

        // amount validation
        if (isset($data['amount']) && $data['amount'] < 0) {
            $errors['amount'] = 'Amount cannot be negative.';
        }

        // tax percentage validation
        if (isset($data['tax_percentage']) && ($data['tax_percentage'] < 0 || $data['tax_percentage'] > 100)) {
            $errors['tax_percentage'] = 'Tax percentage must be between 0 and 100.';
        }

        // charge_type validation
        $validTypes = ['Consultation', 'Room', 'Operation', 'Test', 'Procedure', 'Medicine', 'Other'];
        if (isset($data['charge_type']) && !in_array($data['charge_type'], $validTypes, true)) {
            $errors['charge_type'] = 'Invalid charge type.';
        }

        // duplicate name/code checks
        try {
            $model = new Charge();
            $db = $model->db();
            if (!empty($data['name'])) {
                $sql = 'SELECT id FROM charges WHERE name = :name LIMIT 1';
                $stmt = $db->prepare($sql);
                $stmt->execute([':name' => $data['name']]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($row && (!isset($row['id']) || (int)$row['id'] !== (int)$excludeId)) {
                    $errors['name'] = 'Charge name already exists.';
                }
            }
            if (!empty($data['code'])) {
                $sql = 'SELECT id FROM charges WHERE code = :code LIMIT 1';
                $stmt = $db->prepare($sql);
                $stmt->execute([':code' => $data['code']]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($row && (!isset($row['id']) || (int)$row['id'] !== (int)$excludeId)) {
                    $errors['code'] = 'Charge code already exists.';
                }
            }
        } catch (Throwable $e) {
            // ignore db check errors, they will be surfaced on save
        }

        return $errors;
    }
}






