<?php
declare(strict_types=1);

class DepartmentsController extends Controller
{
    public function index(): void
    {
        // list departments (search, filter, pagination)
        $q = trim((string)($_GET['q'] ?? ''));
        $status = isset($_GET['status']) && $_GET['status'] !== '' ? (int)$_GET['status'] : null;
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $hospitalId = $this->getHospitalId();
        $departments = [];
        $total = 0;
        try {
            $model = new Department();
            $filters = ['q' => $q, 'status' => $status, 'limit' => $perPage, 'offset' => $offset];
            if ($hospitalId !== null) {
                $filters['hospital_id'] = $hospitalId;
            }
            $departments = $model->list($filters);
            $total = $model->count($filters);
        } catch (Throwable $e) {
            $departments = [];
            $total = 0;
        }

        $this->view('departments/departments', [
            'title' => 'Departments',
            'departments' => $departments,
            'q' => $q,
            'status' => $status,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
        ]);
    }

    public function create(): void
    {
        // show add form
        $hospitalId = $this->getHospitalId();
        $doctors = [];
        try {
            $docModel = new Doctor();
            $doctors = $docModel->all($hospitalId);
        } catch (Throwable $e) {
            $doctors = [];
        }
        $this->view('departments/add', ['title' => 'Add Department', 'doctors' => $doctors]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        unset($_SESSION['form_error'], $_SESSION['old']);

        $hospitalId = $this->getHospitalId();
        $data = [
            'hospital_id' => $hospitalId,
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? ''),
            'hod_id' => (int)($_POST['hod_id'] ?? 0) ?: null,
            'description' => trim($_POST['description'] ?? ''),
            'phone' => trim($_POST['phone'] ?? ''),
            'email' => trim($_POST['email'] ?? ''),
            'floor' => trim($_POST['floor'] ?? ''),
            'opd_start' => $_POST['opd_start'] ?? null,
            'opd_end' => $_POST['opd_end'] ?? null,
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];

        // validate input
        $errors = $this->validateDepartment($data);
        if (!empty($errors)) {
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/departments/add');
            exit;
        }

        try {
            $model = new Department();
            $id = $model->create($data);
            $_SESSION['flash_success'] = 'Department created successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/departments/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            $_SESSION['old'] = $data;
            $msg = 'Could not save department.';
            if ($e instanceof PDOException) {
                $info = $e->errorInfo ?? null;
                if (is_array($info) && isset($info[1]) && intval($info[1]) === 1062) {
                    $msg = 'Department name already exists.';
                }
            }
            $_SESSION['form_error'] = $msg;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/departments/add');
            exit;
        }
    }

    public function show(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) { http_response_code(400); echo 'Invalid department id'; return; }
        $hospitalId = $this->getHospitalId();
        try {
            $model = new Department();
            $dept = $model->find($id, $hospitalId);
            if (!$dept) { http_response_code(404); echo 'Department not found'; return; }
            $docModel = new Doctor();
            $doctors = $this->getDoctorsByDepartment($id, $docModel);
            $this->view('departments/view', ['title'=>'Department Details','department'=>$dept,'doctors'=>$doctors]);
        } catch (Throwable $e) { http_response_code(500); echo 'Error loading department'; }
    }

    public function edit(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) { http_response_code(400); echo 'Invalid department id'; return; }
        $hospitalId = $this->getHospitalId();
        try {
            $model = new Department();
            $dept = $model->find($id, $hospitalId);
            if (!$dept) { http_response_code(404); echo 'Department not found'; return; }
            $docModel = new Doctor();
            $doctors = $docModel->all($hospitalId);
            $this->view('departments/edit', ['title'=>'Edit Department','department'=>$dept,'doctors'=>$doctors]);
        } catch (Throwable $e) { http_response_code(500); echo 'Error loading department for edit'; }
    }

    public function update(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid department id'; 
            return; 
        }
        
        // Process and clean input data
        $opdStart = trim($_POST['opd_start'] ?? '');
        $opdEnd = trim($_POST['opd_end'] ?? '');
        
        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? '') ?: null,
            'hod_id' => isset($_POST['hod_id']) && (int)$_POST['hod_id'] > 0 ? (int)$_POST['hod_id'] : null,
            'description' => trim($_POST['description'] ?? '') ?: null,
            'phone' => trim($_POST['phone'] ?? '') ?: null,
            'email' => trim($_POST['email'] ?? '') ?: null,
            'floor' => trim($_POST['floor'] ?? '') ?: null,
            'opd_start' => !empty($opdStart) ? $opdStart : null,
            'opd_end' => !empty($opdEnd) ? $opdEnd : null,
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];
        
        // validate input (exclude current id when checking duplicates)
        $errors = $this->validateDepartment($data, $id);
        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/departments/edit?id=' . urlencode((string)$id));
            exit;
        }
        
        try {
            $model = new Department();
            $result = $model->update($id, $data);
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Department updated successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/departments');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500);
            // Log error for debugging (in production, log to file)
            error_log('Department update error: ' . $e->getMessage());
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['old'] = $data;
            $_SESSION['form_error'] = 'Could not update department: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/departments/edit?id=' . urlencode((string)$id));
            exit;
        }
    }

    /**
     * Validate department input. Returns array of field => message.
     * If $excludeId is provided it will be ignored when checking duplicate name/code.
     */
    protected function validateDepartment(array $data, ?int $excludeId = null): array
    {
        $errors = [];
        // name required
        if (trim((string)($data['name'] ?? '')) === '') {
            $errors['name'] = 'Name is required.';
        } elseif (mb_strlen($data['name']) > 200) {
            $errors['name'] = 'Name is too long (max 200 chars).';
        }

        // code optional but max length
        if (isset($data['code']) && $data['code'] !== '' && mb_strlen($data['code']) > 50) {
            $errors['code'] = 'Code is too long (max 50 chars).';
        }

        // email format
        if (!empty($data['email']) && !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Invalid email address.';
        }

        // phone basic check (digits, spaces, +, -)
        if (!empty($data['phone']) && !preg_match('/^[0-9+()\-\s]{3,20}$/', $data['phone'])) {
            $errors['phone'] = 'Invalid phone number.';
        }

        // OPD times: if both provided, ensure start < end
        if (!empty($data['opd_start']) && !empty($data['opd_end'])) {
            try {
                $s = new DateTime($data['opd_start']);
                $e = new DateTime($data['opd_end']);
                if ($s >= $e) {
                    $errors['opd_start'] = 'OPD start must be before end.';
                    $errors['opd_end'] = 'OPD end must be after start.';
                }
            } catch (Exception $ex) {
                $errors['opd_start'] = 'Invalid OPD time.';
            }
        }

        // duplicate name/code checks
        try {
            $model = new Department();
            $db = $model->db();
            if (!empty($data['name'])) {
                $sql = 'SELECT id FROM departments WHERE name = :name LIMIT 1';
                $stmt = $db->prepare($sql);
                $stmt->execute([':name' => $data['name']]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($row && (!isset($row['id']) || (int)$row['id'] !== (int)$excludeId)) {
                    $errors['name'] = 'Department name already exists.';
                }
            }
            if (!empty($data['code'])) {
                $sql = 'SELECT id FROM departments WHERE code = :code LIMIT 1';
                $stmt = $db->prepare($sql);
                $stmt->execute([':code' => $data['code']]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($row && (!isset($row['id']) || (int)$row['id'] !== (int)$excludeId)) {
                    $errors['code'] = 'Department code already exists.';
                }
            }
        } catch (Throwable $e) {
            // ignore db check errors, they will be surfaced on save
        }

        return $errors;
    }

    public function delete(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { http_response_code(405); echo 'Method Not Allowed'; return; }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { http_response_code(400); echo 'Invalid id'; return; }
        $hospitalId = $this->getHospitalId();
        try {
            $model = new Department();
            // Verify department belongs to hospital before deleting
            if ($hospitalId !== null) {
                $dept = $model->find($id, $hospitalId);
                if (!$dept) {
                    http_response_code(404);
                    echo 'Department not found';
                    return;
                }
            }
            $stmt = $model->db()->prepare('DELETE FROM departments WHERE id = :id' . ($hospitalId !== null ? ' AND hospital_id = :hospital_id' : ''));
            $params = [':id' => $id];
            if ($hospitalId !== null) {
                $params[':hospital_id'] = $hospitalId;
            }
            $stmt->execute($params);
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/departments');
            exit;
        } catch (Throwable $e) { http_response_code(500); echo 'Could not delete department.'; }
    }

    public function deactivate(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { http_response_code(405); echo 'Method Not Allowed'; return; }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { http_response_code(400); echo 'Invalid id'; return; }
        try {
            $model = new Department();
            $stmt = $model->db()->prepare('UPDATE departments SET status = 0 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Department deactivated.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/departments');
            exit;
        } catch (Throwable $e) { http_response_code(500); echo 'Could not deactivate department.'; }
    }

    public function restore(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { http_response_code(405); echo 'Method Not Allowed'; return; }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { http_response_code(400); echo 'Invalid id'; return; }
        try {
            $model = new Department();
            $stmt = $model->db()->prepare('UPDATE departments SET status = 1 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Department restored.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/departments');
            exit;
        } catch (Throwable $e) { http_response_code(500); echo 'Could not restore department.'; }
    }

    protected function getDoctorsByDepartment(int $deptId, Doctor $docModel): array
    {
        $all = $docModel->all();
        $out = [];
        foreach ($all as $d) {
            if (isset($d['department_id']) && (int)$d['department_id'] === $deptId) {
                $out[] = $d;
            }
        }
        return $out;
    }

    public function checkName(): void
    {
        $name = trim((string)($_GET['name'] ?? ''));
        $exists = false;

        if (!empty($name)) {
            try {
                $model = new Department();
                $db = $model->db();
                $sql = 'SELECT id FROM departments WHERE name = :name LIMIT 1';
                $stmt = $db->prepare($sql);
                $stmt->execute([':name' => $name]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                $exists = $row !== false;
            } catch (Throwable $e) {
                // Ignore errors, assume name doesn't exist
            }
        }

        header('Content-Type: application/json');
        echo json_encode(['exists' => $exists]);
        exit;
    }
}
