<?php
declare(strict_types=1);

class DoctorsController extends Controller
{
    public function index(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        $flashSuccess = $_SESSION['flash_success'] ?? null;
        if (isset($_SESSION['flash_success'])) { unset($_SESSION['flash_success']); }

        $q = trim((string)($_GET['q'] ?? ''));
        $department = isset($_GET['department']) ? (int)$_GET['department'] : null;
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $hospitalId = $this->getHospitalId();
        
        $model = new Doctor();
        $filters = ['q' => $q, 'department_id' => $department, 'limit' => $perPage, 'offset' => $offset];
        if ($hospitalId !== null) {
            $filters['hospital_id'] = $hospitalId;
        }
        $doctors = $model->list($filters);
        $total = $model->count($filters);

        $this->view('doctors/list', [
            'title' => 'Doctors',
            'doctors' => $doctors,
            'departments' => (new Department())->all(),
            'q' => $q,
            'department' => $department,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
            'flashSuccess' => $flashSuccess,
        ]);
    }

    public function scheduleCalendar(): void
    {
        $doctorId = isset($_GET['doctor_id']) ? (int)$_GET['doctor_id'] : null;
        $model = new Doctor();
        $schedules = $model->getSchedules($doctorId);

        $this->view('doctors/schedule_calendar', [
            'title' => 'Doctor Schedule Calendar',
            'schedules' => $schedules,
            'selectedDoctor' => $doctorId,
            'doctors' => $model->all($this->getHospitalId()),
        ]);
    }

    public function add(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        $formError = $_SESSION['form_error'] ?? null;
        $formErrors = $_SESSION['form_errors'] ?? [];
        $old = $_SESSION['old'] ?? null;
        if (isset($_SESSION['form_error'])) { unset($_SESSION['form_error']); }
        if (isset($_SESSION['form_errors'])) { unset($_SESSION['form_errors']); }
        if (isset($_SESSION['old'])) { unset($_SESSION['old']); }

        $model = new Doctor();
        $this->view('doctors/add', [
            'title' => 'Add Doctor',
            'departments' => $model->departments(),
            'formError' => $formError,
            'formErrors' => $formErrors,
            'old' => $old,
        ]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $hospitalId = $this->getHospitalId();

        $data = [
            'hospital_id' => $hospitalId,
            'first_name' => trim($_POST['first_name'] ?? ''),
            'last_name'  => trim($_POST['last_name'] ?? ''),
            'specialty'  => trim($_POST['specialty'] ?? ''),
            'department_id' => (int)($_POST['department_id'] ?? 0) ?: null,
            'phone'      => trim($_POST['phone'] ?? ''),
            'email'      => trim($_POST['email'] ?? ''),
        ];

        $errors = $this->validateDoctor($data);

        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/doctors/add');
            exit;
        }

        try {
            $model = new Doctor();
            $id = $model->create($data);

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Doctor added successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/doctors/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            error_log('Doctor creation error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not add doctor: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/doctors/add');
            exit;
        }
    }

    public function attendance(): void
    {
        $doctorId = isset($_GET['doctor_id']) ? (int)$_GET['doctor_id'] : null;
        $dateFrom = $_GET['date_from'] ?? null;
        $dateTo   = $_GET['date_to'] ?? null;
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $model = new Doctor();
        $items = $model->listAttendance([
            'doctor_id' => $doctorId,
            'date_from' => $dateFrom,
            'date_to'   => $dateTo,
            'limit'     => $perPage,
            'offset'    => $offset,
        ]);
        $total = $model->countAttendance([
            'doctor_id' => $doctorId,
            'date_from' => $dateFrom,
            'date_to'   => $dateTo,
        ]);

        $this->view('doctors/attendance', [
            'title' => 'Doctor Attendance',
            'items' => $items,
            'doctors' => $model->all($this->getHospitalId()),
            'selectedDoctor' => $doctorId,
            'dateFrom' => $dateFrom,
            'dateTo' => $dateTo,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
        ]);
    }

    public function attendanceStore(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $data = [
            'doctor_id' => (int)($_POST['doctor_id'] ?? 0),
            'att_date' => trim($_POST['att_date'] ?? ''),
            'status' => trim($_POST['status'] ?? 'Full Day'),
            'check_in' => trim($_POST['check_in'] ?? ''),
            'check_out' => trim($_POST['check_out'] ?? ''),
            'overtime_minutes' => (int)($_POST['overtime_minutes'] ?? 0),
            'remarks' => trim($_POST['remarks'] ?? ''),
        ];

        $errors = [];
        if ($data['doctor_id'] <= 0) { $errors[] = 'Doctor is required.'; }
        if ($data['att_date'] === '') { $errors[] = 'Date is required.'; }
        $allowedStatus = ['Full Day','Half Day','Absent','Leave'];
        if (!in_array($data['status'], $allowedStatus, true)) { $errors[] = 'Invalid status.'; }
        if ($data['overtime_minutes'] < 0) { $errors[] = 'Overtime cannot be negative.'; }

        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = implode(' ', $errors);
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/doctors/attendance');
            exit;
        }

        try {
            $model = new Doctor();
            $model->createAttendance($data);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Attendance added successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/doctors/attendance');
            exit;
        } catch (Throwable $e) {
            error_log('Attendance create error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not add attendance: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/doctors/attendance');
            exit;
        }
    }

    public function attendanceDelete(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) {
            http_response_code(422);
            echo 'Invalid id';
            return;
        }
        try {
            (new Doctor())->deleteAttendance($id);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Attendance deleted successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/doctors/attendance');
            exit;
        } catch (Throwable $e) {
            error_log('Attendance delete error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not delete attendance: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/doctors/attendance');
            exit;
        }
    }

    public function show(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid doctor id';
            return;
        }
        $model = new Doctor();
        $doctor = $model->find($id);
        if (!$doctor) {
            http_response_code(404);
            echo 'Doctor not found';
            return;
        }
        $this->view('doctors/view', [
            'title' => 'Doctor Details',
            'doctor' => $doctor,
        ]);
    }

    public function schedule(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        
        // MVC Pattern: Controller handles session, passes data to view
        $flashSuccess = $_SESSION['flash_success'] ?? null;
        $formError = $_SESSION['form_error'] ?? null;
        
        // Clear flash messages after reading
        if (isset($_SESSION['flash_success'])) { unset($_SESSION['flash_success']); }
        if (isset($_SESSION['form_error'])) { unset($_SESSION['form_error']); }
        
        $doctorId = isset($_GET['doctor_id']) ? (int)$_GET['doctor_id'] : null;
        $model = new Doctor();
        $schedules = $model->getSchedules($doctorId);
        $doctors = $model->all();

        $this->view('doctors/schedule', [
            'title' => 'Doctor Schedule',
            'schedules' => $schedules,
            'doctors' => $doctors,
            'selectedDoctor' => $doctorId,
            'flashSuccess' => $flashSuccess,
            'formError' => $formError,
        ]);
    }

    public function scheduleStore(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $data = [
            'doctor_id' => (int)($_POST['doctor_id'] ?? 0),
            'day_of_week' => trim($_POST['day_of_week'] ?? ''),
            'start_time' => trim($_POST['start_time'] ?? ''),
            'end_time' => trim($_POST['end_time'] ?? ''),
            'is_available' => isset($_POST['is_available']) ? 1 : 0,
        ];

        $errors = $this->validateSchedule($data);

        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = implode(' ', $errors);
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/doctors/schedule?doctor_id=' . urlencode((string)$data['doctor_id']));
            exit;
        }

        try {
            $model = new Doctor();
            $model->createSchedule($data);

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Schedule added successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/doctors/schedule?doctor_id=' . urlencode((string)$data['doctor_id']));
            exit;
        } catch (Throwable $e) {
            error_log('Schedule creation error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not add schedule: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/doctors/schedule?doctor_id=' . urlencode((string)$data['doctor_id']));
            exit;
        }
    }

    public function scheduleDelete(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) {
            http_response_code(422);
            echo 'Invalid schedule id';
            return;
        }

        try {
            $model = new Doctor();
            $model->deleteSchedule($id);

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Schedule deleted successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/doctors/schedule');
            exit;
        } catch (Throwable $e) {
            error_log('Schedule deletion error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not delete schedule: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/doctors/schedule');
            exit;
        }
    }

    protected function validateDoctor(array $data): array
    {
        $errors = [];

        if (empty($data['first_name'])) {
            $errors['first_name'] = 'First name is required.';
        }
        if (empty($data['last_name'])) {
            $errors['last_name'] = 'Last name is required.';
        }
        if (!empty($data['email']) && !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Invalid email format.';
        }

        return $errors;
    }

    protected function validateSchedule(array $data): array
    {
        $errors = [];

        if (empty($data['doctor_id']) || $data['doctor_id'] <= 0) {
            $errors[] = 'Doctor is required.';
        }
        if (empty($data['day_of_week'])) {
            $errors[] = 'Day of week is required.';
        }
        if (empty($data['start_time'])) {
            $errors[] = 'Start time is required.';
        }
        if (empty($data['end_time'])) {
            $errors[] = 'End time is required.';
        }
        if (!empty($data['start_time']) && !empty($data['end_time']) && $data['start_time'] >= $data['end_time']) {
            $errors[] = 'End time must be after start time.';
        }

        return $errors;
    }
}
