<?php
declare(strict_types=1);

class OPDController extends Controller
{
    public function registration(): void
    {
        $patientId = isset($_GET['patient_id']) ? (int)$_GET['patient_id'] : 0;
        
        $patients = [];
        $doctors = [];
        $departments = [];
        
        try {
            $appointmentModel = new Appointment();
            $patients = $appointmentModel->patients();
            
            $doctorModel = new Doctor();
            $doctors = $doctorModel->all();
            
            $deptModel = new Department();
            $departments = $deptModel->all();
            
            // Get patient details if patient_id is provided
            $patient = null;
            if ($patientId > 0) {
                $patientModel = new Patient();
                $patient = $patientModel->find($patientId);
            }
        } catch (Throwable $e) {
            $patients = [];
            $doctors = [];
            $departments = [];
            $patient = null;
        }

        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        $formError = $_SESSION['form_error'] ?? null;
        $formErrors = $_SESSION['form_errors'] ?? [];
        $old = $_SESSION['old'] ?? null;
        if (isset($_SESSION['form_error'])) { unset($_SESSION['form_error']); }
        if (isset($_SESSION['form_errors'])) { unset($_SESSION['form_errors']); }
        if (isset($_SESSION['old'])) { unset($_SESSION['old']); }

        $this->view('opd/registration', [
            'title' => 'OPD Registration',
            'patients' => $patients,
            'doctors' => $doctors,
            'departments' => $departments,
            'patient' => $patient,
            'patient_id' => $patientId,
            'formError' => $formError,
            'formErrors' => $formErrors,
            'old' => $old,
        ]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $data = [
            'patient_id' => isset($_POST['patient_id']) ? (int)$_POST['patient_id'] : 0,
            'doctor_id' => isset($_POST['doctor_id']) ? (int)$_POST['doctor_id'] : 0,
            'scheduled_at' => trim($_POST['scheduled_at'] ?? ''),
            'status' => trim($_POST['status'] ?? 'Booked'),
            'notes' => trim($_POST['notes'] ?? '') ?: null,
        ];

        $errors = $this->validateOPDRegistration($data);
        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/opd/registration' . ($data['patient_id'] > 0 ? '?patient_id=' . urlencode((string)$data['patient_id']) : ''));
            exit;
        }

        try {
            $model = new Appointment();
            $id = $model->create($data);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'OPD registration successful. Appointment booked.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/opd/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_error'] = 'Could not register OPD: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/opd/registration' . ($data['patient_id'] > 0 ? '?patient_id=' . urlencode((string)$data['patient_id']) : ''));
            exit;
        }
    }

    public function index(): void
    {
        $q = trim((string)($_GET['q'] ?? ''));
        $doctorId = isset($_GET['doctor_id']) && (int)$_GET['doctor_id'] > 0 ? (int)$_GET['doctor_id'] : null;
        $status = isset($_GET['status']) && $_GET['status'] !== '' ? trim($_GET['status']) : null;
        $date = isset($_GET['date']) && $_GET['date'] !== '' ? trim($_GET['date']) : date('Y-m-d');
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $appointments = [];
        $total = 0;
        $doctors = [];
        
        try {
            $model = new Appointment();
            $db = $model->db();
            
            $sql = "SELECT 
                        a.id,
                        a.scheduled_at,
                        a.status,
                        a.notes,
                        a.created_at,
                        p.id AS patient_id,
                        p.first_name AS p_first,
                        p.last_name AS p_last,
                        p.phone AS p_phone,
                        p.gender AS p_gender,
                        d.id AS doctor_id,
                        d.first_name AS d_first,
                        d.last_name AS d_last,
                        d.specialty,
                        dept.name AS department_name
                    FROM appointments a
                    INNER JOIN patients p ON p.id = a.patient_id
                    INNER JOIN doctors d ON d.id = a.doctor_id
                    LEFT JOIN departments dept ON dept.id = d.department_id
                    WHERE 1=1";
            $params = [];

            if ($q !== '') {
                $sql .= " AND (p.first_name LIKE :q OR p.last_name LIKE :q OR p.phone LIKE :q)";
                $params[':q'] = '%' . $q . '%';
            }

            if ($doctorId !== null) {
                $sql .= " AND d.id = :doctor_id";
                $params[':doctor_id'] = $doctorId;
            }

            if ($status !== null) {
                $sql .= " AND a.status = :status";
                $params[':status'] = $status;
            }

            if ($date) {
                $sql .= " AND DATE(a.scheduled_at) = :date";
                $params[':date'] = $date;
            }

            $sql .= " ORDER BY a.scheduled_at DESC LIMIT :limit OFFSET :offset";
            $stmt = $db->prepare($sql);
            foreach ($params as $key => $value) {
                $stmt->bindValue($key, $value);
            }
            $stmt->bindValue(':limit', (int)$perPage, PDO::PARAM_INT);
            $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
            $stmt->execute();
            $appointments = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Count total
            $countSql = "SELECT COUNT(*) 
                        FROM appointments a
                        INNER JOIN patients p ON p.id = a.patient_id
                        INNER JOIN doctors d ON d.id = a.doctor_id
                        WHERE 1=1";
            $countParams = [];
            if ($q !== '') {
                $countSql .= " AND (p.first_name LIKE :q OR p.last_name LIKE :q OR p.phone LIKE :q)";
                $countParams[':q'] = '%' . $q . '%';
            }
            if ($doctorId !== null) {
                $countSql .= " AND d.id = :doctor_id";
                $countParams[':doctor_id'] = $doctorId;
            }
            if ($status !== null) {
                $countSql .= " AND a.status = :status";
                $countParams[':status'] = $status;
            }
            if ($date) {
                $countSql .= " AND DATE(a.scheduled_at) = :date";
                $countParams[':date'] = $date;
            }
            $countStmt = $db->prepare($countSql);
            foreach ($countParams as $key => $value) {
                $countStmt->bindValue($key, $value);
            }
            $countStmt->execute();
            $total = (int)$countStmt->fetchColumn();

            $doctorModel = new Doctor();
            $doctors = $doctorModel->all();
        } catch (Throwable $e) {
            $appointments = [];
            $total = 0;
            $doctors = [];
        }

        $this->view('opd/list', [
            'title' => 'OPD Registrations',
            'appointments' => $appointments,
            'doctors' => $doctors,
            'q' => $q,
            'doctor_id' => $doctorId,
            'status' => $status,
            'date' => $date,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
        ]);
    }

    public function show(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid appointment id';
            return;
        }

        try {
            $model = new Appointment();
            $db = $model->db();
            $sql = "SELECT 
                        a.*,
                        p.id AS patient_id,
                        p.first_name AS p_first,
                        p.last_name AS p_last,
                        p.phone AS p_phone,
                        p.email AS p_email,
                        p.gender AS p_gender,
                        p.dob AS p_dob,
                        p.address AS p_address,
                        d.id AS doctor_id,
                        d.first_name AS d_first,
                        d.last_name AS d_last,
                        d.specialty,
                        d.phone AS d_phone,
                        d.email AS d_email,
                        dept.name AS department_name,
                        dept.code AS department_code
                    FROM appointments a
                    INNER JOIN patients p ON p.id = a.patient_id
                    INNER JOIN doctors d ON d.id = a.doctor_id
                    LEFT JOIN departments dept ON dept.id = d.department_id
                    WHERE a.id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute([':id' => $id]);
            $appointment = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$appointment) {
                http_response_code(404);
                echo 'Appointment not found';
                return;
            }

            $this->view('opd/view', [
                'title' => 'OPD Registration Details',
                'appointment' => $appointment,
            ]);
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Error loading appointment';
        }
    }

    protected function validateOPDRegistration(array $data): array
    {
        $errors = [];

        if ($data['patient_id'] <= 0) {
            $errors['patient_id'] = 'Patient is required.';
        }

        if ($data['doctor_id'] <= 0) {
            $errors['doctor_id'] = 'Doctor is required.';
        }

        if (trim($data['scheduled_at'] ?? '') === '') {
            $errors['scheduled_at'] = 'Appointment date and time is required.';
        } else {
            $scheduledAt = strtotime($data['scheduled_at']);
            if ($scheduledAt === false) {
                $errors['scheduled_at'] = 'Invalid date/time format.';
            } elseif ($scheduledAt < time()) {
                // Allow past dates for walk-in patients
                // $errors['scheduled_at'] = 'Appointment date cannot be in the past.';
            }
        }

        return $errors;
    }
}

