<?php
declare(strict_types=1);

class PatientsController extends Controller
{
    public function index(): void
    {
        $q = trim((string)($_GET['q'] ?? ''));
        $gender = isset($_GET['gender']) && $_GET['gender'] !== '' ? trim($_GET['gender']) : null;
        $patientType = isset($_GET['patient_type']) && $_GET['patient_type'] !== '' ? trim($_GET['patient_type']) : null;
        $status = isset($_GET['status']) && $_GET['status'] !== '' ? trim($_GET['status']) : null;
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $hospitalId = $this->getHospitalId();
        
        $patients = [];
        $total = 0;
        try {
            $model = new Patient();
            $filters = [
                'q' => $q, 
                'gender' => $gender, 
                'patient_type' => $patientType,
                'status' => $status,
                'limit' => $perPage, 
                'offset' => $offset
            ];
            if ($hospitalId !== null) {
                $filters['hospital_id'] = $hospitalId;
            }
            $patients = $model->list($filters);
            $total = $model->count($filters);
        } catch (Throwable $e) {
            $patients = [];
            $total = 0;
        }

        $this->view('patients/list', [
            'title' => 'Patients',
            'patients' => $patients,
            'q' => $q,
            'gender' => $gender,
            'patient_type' => $patientType,
            'status' => $status,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
        ]);
    }

    public function add(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        $formError = $_SESSION['form_error'] ?? null;
        $formErrors = $_SESSION['form_errors'] ?? [];
        $old = $_SESSION['old'] ?? null;
        if (isset($_SESSION['form_error'])) { unset($_SESSION['form_error']); }
        if (isset($_SESSION['form_errors'])) { unset($_SESSION['form_errors']); }
        if (isset($_SESSION['old'])) { unset($_SESSION['old']); }

        $this->view('patients/add', [
            'title' => 'Add Patient',
            'formError' => $formError,
            'formErrors' => $formErrors,
            'old' => $old,
        ]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $hospitalId = $this->getHospitalId();

        $data = [
            'hospital_id' => $hospitalId,
            'first_name' => trim($_POST['first_name'] ?? ''),
            'last_name'  => trim($_POST['last_name'] ?? ''),
            'gender'     => trim($_POST['gender'] ?? '') ?: null,
            'dob'        => trim($_POST['dob'] ?? '') ?: null,
            'phone'      => trim($_POST['phone'] ?? '') ?: null,
            'email'      => trim($_POST['email'] ?? '') ?: null,
            'address'    => trim($_POST['address'] ?? '') ?: null,
        ];

        $errors = $this->validatePatient($data);
        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/patients/add');
            exit;
        }

        try {
            $model = new Patient();
            $id = $model->create($data);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Patient added successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/patients/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_error'] = 'Could not add patient: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/patients/add');
            exit;
        }
    }

    public function show(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid patient id';
            return;
        }
        $hospitalId = $this->getHospitalId();
        
        $model = new Patient();
        $patient = $model->find($id, $hospitalId);
        if (!$patient) {
            http_response_code(404);
            echo 'Patient not found';
            return;
        }
        $this->view('patients/view', [
            'title' => 'Patient Details',
            'patient' => $patient,
        ]);
    }

    public function edit(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid patient id';
            return;
        }

        $hospitalId = $this->getHospitalId();
        
        $model = new Patient();
        $patient = $model->find($id, $hospitalId);
        if (!$patient) {
            http_response_code(404);
            echo 'Patient not found';
            return;
        }

        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        $formError = $_SESSION['form_error'] ?? null;
        $formErrors = $_SESSION['form_errors'] ?? [];
        $old = $_SESSION['old'] ?? null;
        if (isset($_SESSION['form_error'])) { unset($_SESSION['form_error']); }
        if (isset($_SESSION['form_errors'])) { unset($_SESSION['form_errors']); }
        if (isset($_SESSION['old'])) { unset($_SESSION['old']); }

        $this->view('patients/edit', [
            'title' => 'Edit Patient',
            'patient' => $patient,
            'formError' => $formError,
            'formErrors' => $formErrors,
            'old' => $old,
        ]);
    }

    public function update(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid patient id';
            return;
        }

        $data = [
            'first_name' => trim($_POST['first_name'] ?? ''),
            'last_name'  => trim($_POST['last_name'] ?? ''),
            'gender'     => trim($_POST['gender'] ?? '') ?: null,
            'dob'        => trim($_POST['dob'] ?? '') ?: null,
            'phone'      => trim($_POST['phone'] ?? '') ?: null,
            'email'      => trim($_POST['email'] ?? '') ?: null,
            'address'    => trim($_POST['address'] ?? '') ?: null,
        ];

        $errors = $this->validatePatient($data);
        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/patients/edit?id=' . urlencode((string)$id));
            exit;
        }

        try {
            $model = new Patient();
            $model->update($id, $data);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Patient updated successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/patients');
            exit;
        } catch (Throwable $e) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_error'] = 'Could not update patient: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/patients/edit?id=' . urlencode((string)$id));
            exit;
        }
    }

    public function delete(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid patient id';
            return;
        }

        try {
            $model = new Patient();
            $stmt = $model->db()->prepare('DELETE FROM patients WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Patient deleted successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/patients');
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not delete patient.';
        }
    }

    public function print(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid patient id';
            return;
        }

        $model = new Patient();
        $patient = $model->find($id);
        if (!$patient) {
            http_response_code(404);
            echo 'Patient not found';
            return;
        }

        $this->view('patients/print', [
            'title' => 'Patient Card',
            'patient' => $patient,
        ]);
    }

    protected function validatePatient(array $data): array
    {
        $errors = [];

        if (trim((string)($data['first_name'] ?? '')) === '') {
            $errors['first_name'] = 'First name is required.';
        }

        if (trim((string)($data['last_name'] ?? '')) === '') {
            $errors['last_name'] = 'Last name is required.';
        }

        if (!empty($data['email']) && !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Invalid email format.';
        }

        if (!empty($data['phone']) && !preg_match('/^[0-9+\-\s()]+$/', $data['phone'])) {
            $errors['phone'] = 'Invalid phone number format.';
        }

        return $errors;
    }

    public function dischargeSummary(): void
    {
        $this->view('patients/discharge_summary', [
            'title' => 'Discharge Summary',
        ]);
    }

    public function addDischargeSummary(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $data = [
            'patient_name'   => trim($_POST['patientName'] ?? ''),
            'patient_id'     => trim($_POST['patientID'] ?? ''),
            'admission_date' => trim($_POST['admissionDate'] ?? ''),
            'discharge_date' => trim($_POST['dischargeDate'] ?? ''),
            'diagnosis'      => trim($_POST['diagnosis'] ?? ''),
            'treatment'      => trim($_POST['treatment'] ?? ''),
            'medications'    => trim($_POST['medications'] ?? ''),
            'follow_up'      => trim($_POST['followUp'] ?? ''),
        ];

        $errors = [];
        if ($data['patient_name'] === '') { $errors['patient_name'] = 'Patient name is required.'; }
        if ($data['patient_id'] === '') { $errors['patient_id'] = 'Patient ID is required.'; }
        if ($data['admission_date'] === '') { $errors['admission_date'] = 'Admission date is required.'; }
        if ($data['discharge_date'] === '') { $errors['discharge_date'] = 'Discharge date is required.'; }
        if ($data['diagnosis'] === '') { $errors['diagnosis'] = 'Diagnosis is required.'; }
        if ($data['treatment'] === '') { $errors['treatment'] = 'Treatment is required.'; }
        if ($data['medications'] === '') { $errors['medications'] = 'Medications are required.'; }

        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/patients/discharge-summary');
            exit;
        }

        try {
            $model = new DischargeSummary();
            $model->create($data);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Discharge summary saved successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/patients/discharge-summary');
            exit;
        } catch (Throwable $e) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_error'] = 'Could not save discharge summary: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/patients/discharge-summary');
            exit;
        }
    }

    public function discharged(): void
    {
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        try {
            $admission = new Admission();
            $items = $admission->list([
                'status' => 'Discharged',
                'limit' => $perPage,
                'offset' => $offset,
            ]);
            $total = $admission->count([
                'status' => 'Discharged',
            ]);
        } catch (Throwable $e) {
            $items = [];
            $total = 0;
        }

        $this->view('patients/discharged', [
            'title' => 'Discharged Patients',
            'items' => $items,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
        ]);
    }
}
