<?php
declare(strict_types=1);

class PharmacyController extends Controller
{
    // Medicines
    public function medicines(): void
    {
        $med = new Medicine();
        $this->view('pharmacy/medicines', [
            'title' => 'Medicines',
            'medicines' => $med->all(),
        ]);
    }

    public function addMedicine(): void
    {
        $this->view('pharmacy/add_medicine', [
            'title' => 'Add Medicine',
        ]);
    }

    public function storeMedicine(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'sku' => trim($_POST['sku'] ?? ''),
            'unit' => trim($_POST['unit'] ?? ''),
            'price' => (float)($_POST['price'] ?? 0),
        ];
        if ($data['name'] === '') {
            http_response_code(422);
            echo 'Name is required';
            return;
        }
        (new Medicine())->create($data);
        header('Location: /admin/pharmacy/medicines');
        exit;
    }

    // Stock
    public function stock(): void
    {
        $stock = new StockMovement();
        $this->view('pharmacy/stock', [
            'title' => 'Pharmacy Stock',
            'medicines' => $stock->medicines(),
            'stock_levels' => $stock->stockLevels(),
            'movements' => $stock->all(),
        ]);
    }

    public function stockIn(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $mid = (int)($_POST['medicine_id'] ?? 0);
        $qty = (float)($_POST['qty'] ?? 0);
        $ref = trim($_POST['ref'] ?? '');
        if ($mid <= 0 || $qty <= 0) {
            http_response_code(422);
            echo 'Valid medicine and quantity required';
            return;
        }
        (new StockMovement())->addIn($mid, $qty, $ref ?: null);
        header('Location: /admin/pharmacy/stock');
        exit;
    }

    public function stockOut(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $mid = (int)($_POST['medicine_id'] ?? 0);
        $qty = (float)($_POST['qty'] ?? 0);
        $ref = trim($_POST['ref'] ?? '');
        if ($mid <= 0 || $qty <= 0) {
            http_response_code(422);
            echo 'Valid medicine and quantity required';
            return;
        }
        (new StockMovement())->addOut($mid, $qty, $ref ?: null);
        header('Location: /admin/pharmacy/stock');
        exit;
    }

    // Sales
    public function sales(): void
    {
        $sale = new Sale();
        $this->view('pharmacy/sales', [
            'title' => 'Pharmacy Sales',
            'sales' => $sale->all(),
        ]);
    }

    public function addSale(): void
    {
        $sale = new Sale();
        $stock = new StockMovement();
        $this->view('pharmacy/add_sale', [
            'title' => 'New Sale',
            'patients' => $sale->patients(),
            'medicines' => $stock->medicines(),
        ]);
    }

    public function storeSale(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $patientId = (int)($_POST['patient_id'] ?? 0) ?: null;
        $items = $_POST['items'] ?? [];
        if (empty($items)) {
            http_response_code(422);
            echo 'At least one item is required';
            return;
        }
        // Compute total using submitted unit_price; fallback to medicines.price
        $medModel = new Medicine();
        $total = 0.0;
        foreach ($items as $it) {
            $mid = (int)($it['medicine_id'] ?? 0);
            $qty = (float)($it['qty'] ?? 0);
            if ($mid <= 0 || $qty <= 0) { continue; }
            $price = isset($it['unit_price']) ? (float)$it['unit_price'] : 0.0;
            if ($price <= 0) {
                $price = $medModel->getPrice($mid);
            }
            $total += $qty * max(0, $price);
        }
        $saleId = (new Sale())->create($patientId, $total);
        // Record stock OUT movements with ref
        $stock = new StockMovement();
        foreach ($items as $it) {
            $mid = (int)($it['medicine_id'] ?? 0);
            $qty = (float)($it['qty'] ?? 0);
            if ($mid > 0 && $qty > 0) {
                $stock->addOut($mid, $qty, 'SALE:#' . $saleId);
            }
        }
        header('Location: /admin/pharmacy/sales/view?id=' . urlencode((string)$saleId));
        exit;
    }

    public function viewSale(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid sale id';
            return;
        }
        $sale = (new Sale())->find($id);
        if (!$sale) {
            http_response_code(404);
            echo 'Sale not found';
            return;
        }
        // Load movements tagged with this sale as line details
        $items = (new Medicine())->saleItemsByRef('SALE:#' . $id);
        $this->view('pharmacy/sale_view', [
            'title' => 'Sale #' . $id,
            'sale' => $sale,
            'items' => $items,
        ]);
    }
}
