<?php
declare(strict_types=1);

class RoomsController extends Controller
{
    public function index(): void
    {
        // list rooms (search, filter, pagination)
        $q = trim((string)($_GET['q'] ?? ''));
        $status = isset($_GET['status']) && $_GET['status'] !== '' ? (int)$_GET['status'] : null;
        $roomType = isset($_GET['room_type']) && $_GET['room_type'] !== '' ? (string)$_GET['room_type'] : null;
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $hospitalId = $this->getHospitalId();
        $rooms = [];
        $total = 0;
        $roomTypes = [];
        try {
            $model = new Room();
            $filters = [
                'q' => $q, 
                'status' => $status, 
                'room_type' => $roomType, 
                'limit' => $perPage, 
                'offset' => $offset
            ];
            if ($hospitalId !== null) {
                $filters['hospital_id'] = $hospitalId;
            }
            $rooms = $model->list($filters);
            $total = $model->count($filters);
            
            $roomTypes = $model->getRoomTypes();
        } catch (Throwable $e) {
            $rooms = [];
            $total = 0;
            $roomTypes = [];
        }

        $this->view('rooms/list', [
            'title' => 'Rooms & Wards Management',
            'rooms' => $rooms,
            'roomTypes' => $roomTypes,
            'q' => $q,
            'status' => $status,
            'room_type' => $roomType,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
        ]);
    }

    public function create(): void
    {
        $roomTypes = [];
        try {
            $model = new Room();
            $roomTypes = $model->getRoomTypes();
        } catch (Throwable $e) {
            $roomTypes = [];
        }
        $this->view('rooms/add', [
            'title' => 'Add Room/Ward',
            'roomTypes' => $roomTypes,
        ]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        if (session_status() !== PHP_SESSION_ACTIVE) { 
            session_start(); 
        }
        unset($_SESSION['form_error'], $_SESSION['old']);

        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? '') ?: null,
            'room_type' => trim($_POST['room_type'] ?? 'General'),
            'floor' => trim($_POST['floor'] ?? '') ?: null,
            'building' => trim($_POST['building'] ?? '') ?: null,
            'capacity' => isset($_POST['capacity']) ? (int)$_POST['capacity'] : 1,
            'charge_per_day' => isset($_POST['charge_per_day']) ? (float)$_POST['charge_per_day'] : 0.00,
            'description' => trim($_POST['description'] ?? '') ?: null,
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];

        // validate input
        $errors = $this->validateRoom($data);
        if (!empty($errors)) {
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/rooms/add');
            exit;
        }

        try {
            $model = new Room();
            $id = $model->create($data);
            $_SESSION['flash_success'] = 'Room/Ward created successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/rooms/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            $_SESSION['old'] = $data;
            $msg = 'Could not save room/ward.';
            if ($e instanceof PDOException) {
                $info = $e->errorInfo ?? null;
                if (is_array($info) && isset($info[1]) && intval($info[1]) === 1062) {
                    $msg = 'Room code already exists.';
                }
            }
            $_SESSION['form_error'] = $msg;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/rooms/add');
            exit;
        }
    }

    public function show(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid room id'; 
            return; 
        }
        $hospitalId = $this->getHospitalId();
        try {
            $model = new Room();
            $room = $model->find($id, $hospitalId);
            if (!$room) { 
                http_response_code(404); 
                echo 'Room not found'; 
                return; 
            }
            
            // Get beds for this room
            $beds = $model->getBeds($id);
            
            $this->view('rooms/view', [
                'title' => 'Room/Ward Details',
                'room' => $room,
                'beds' => $beds,
            ]);
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Error loading room'; 
        }
    }

    public function edit(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid room id'; 
            return; 
        }
        $hospitalId = $this->getHospitalId();
        try {
            $model = new Room();
            $room = $model->find($id, $hospitalId);
            if (!$room) { 
                http_response_code(404); 
                echo 'Room not found'; 
                return; 
            }
            $roomTypes = $model->getRoomTypes();
            $this->view('rooms/edit', [
                'title' => 'Edit Room/Ward',
                'room' => $room,
                'roomTypes' => $roomTypes,
            ]);
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Error loading room for edit'; 
        }
    }

    public function update(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid room id'; 
            return; 
        }
        
        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? '') ?: null,
            'room_type' => trim($_POST['room_type'] ?? 'General'),
            'floor' => trim($_POST['floor'] ?? '') ?: null,
            'building' => trim($_POST['building'] ?? '') ?: null,
            'capacity' => isset($_POST['capacity']) ? (int)$_POST['capacity'] : 1,
            'charge_per_day' => isset($_POST['charge_per_day']) ? (float)$_POST['charge_per_day'] : 0.00,
            'description' => trim($_POST['description'] ?? '') ?: null,
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];
        
        // validate input
        $errors = $this->validateRoom($data, $id);
        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/rooms/edit?id=' . urlencode((string)$id));
            exit;
        }
        
        $hospitalId = $this->getHospitalId();
        try {
            $model = new Room();
            // Verify ownership before update
            $room = $model->find($id, $hospitalId);
            if (!$room) {
                http_response_code(404);
                echo 'Room not found';
                return;
            }
            $result = $model->update($id, $data);
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Room/Ward updated successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/rooms');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500);
            error_log('Room update error: ' . $e->getMessage());
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['old'] = $data;
            $_SESSION['form_error'] = 'Could not update room/ward: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/rooms/edit?id=' . urlencode((string)$id));
            exit;
        }
    }

    public function addBed(): void
    {
        $roomId = isset($_GET['room_id']) ? (int)$_GET['room_id'] : 0;
        if ($roomId <= 0) { 
            http_response_code(400); 
            echo 'Invalid room id'; 
            return; 
        }
        try {
            $model = new Room();
            $room = $model->find($roomId);
            if (!$room) { 
                http_response_code(404); 
                echo 'Room not found'; 
                return; 
            }
            $this->view('rooms/add_bed', [
                'title' => 'Add Bed to Room',
                'room' => $room,
            ]);
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Error loading room'; 
        }
    }

    public function storeBed(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        
        $roomId = isset($_POST['room_id']) ? (int)$_POST['room_id'] : 0;
        if ($roomId <= 0) {
            http_response_code(400);
            echo 'Invalid room id';
            return;
        }

        $hospitalId = $this->getHospitalId();
        $data = [
            'hospital_id' => $hospitalId,
            'room_id' => $roomId,
            'ward' => trim($_POST['ward'] ?? ''),
            'number' => trim($_POST['number'] ?? ''),
            'type' => trim($_POST['type'] ?? '') ?: null,
        ];

        if ($data['ward'] === '' || $data['number'] === '') {
            http_response_code(422);
            echo 'Ward and Number are required.';
            return;
        }

        try {
            $bedModel = new Bed();
            $id = $bedModel->create($data);
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Bed added successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/rooms/view?id=' . urlencode((string)$roomId));
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not add bed.';
        }
    }

    public function editBed(): void
    {
        $bedId = isset($_GET['bed_id']) ? (int)$_GET['bed_id'] : 0;
        if ($bedId <= 0) { 
            http_response_code(400); 
            echo 'Invalid bed id'; 
            return; 
        }
        try {
            $bedModel = new Bed();
            $db = $bedModel->db();
            $stmt = $db->prepare("SELECT b.*, r.id AS room_id, r.name AS room_name FROM beds b LEFT JOIN rooms r ON r.id = b.room_id WHERE b.id = :id LIMIT 1");
            $stmt->execute([':id' => $bedId]);
            $bed = $stmt->fetch();
            
            if (!$bed) { 
                http_response_code(404); 
                echo 'Bed not found'; 
                return; 
            }
            
            $this->view('rooms/edit_bed', [
                'title' => 'Edit Bed',
                'bed' => $bed,
            ]);
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Error loading bed for edit'; 
        }
    }

    public function updateBed(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        
        $bedId = isset($_POST['bed_id']) ? (int)$_POST['bed_id'] : 0;
        $roomId = isset($_POST['room_id']) ? (int)$_POST['room_id'] : 0;
        
        if ($bedId <= 0) { 
            http_response_code(400); 
            echo 'Invalid bed id'; 
            return; 
        }

        $ward = trim($_POST['ward'] ?? '');
        $number = trim($_POST['number'] ?? '');
        $type = trim($_POST['type'] ?? '') ?: null;

        if ($ward === '' || $number === '') {
            http_response_code(422);
            echo 'Ward and Number are required.';
            return;
        }

        try {
            $bedModel = new Bed();
            $db = $bedModel->db();
            $stmt = $db->prepare("UPDATE beds SET room_id = :room_id, ward = :ward, number = :number, type = :type WHERE id = :id");
            $stmt->execute([
                ':room_id' => $roomId > 0 ? $roomId : null,
                ':ward' => $ward,
                ':number' => $number,
                ':type' => $type,
                ':id' => $bedId,
            ]);
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Bed updated successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            if ($roomId > 0) {
                header('Location: ' . $base . '/rooms/view?id=' . urlencode((string)$roomId));
            } else {
                header('Location: ' . $base . '/beds');
            }
            exit;
        } catch (Throwable $e) { 
            http_response_code(500);
            error_log('Bed update error: ' . $e->getMessage());
            echo 'Could not update bed: ' . $e->getMessage();
        }
    }

    public function deleteBed(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        $bedId = isset($_POST['bed_id']) ? (int)$_POST['bed_id'] : 0;
        $roomId = isset($_POST['room_id']) ? (int)$_POST['room_id'] : 0;
        
        if ($bedId <= 0) { 
            http_response_code(400); 
            echo 'Invalid bed id'; 
            return; 
        }
        try {
            $bedModel = new Bed();
            $db = $bedModel->db();
            
            // Check if bed is occupied
            $checkStmt = $db->prepare("SELECT COUNT(*) FROM admissions WHERE bed_id = :bed_id AND discharged_at IS NULL");
            $checkStmt->execute([':bed_id' => $bedId]);
            $occupied = (int)$checkStmt->fetchColumn();
            
            if ($occupied > 0) {
                if (session_status() !== PHP_SESSION_ACTIVE) { 
                    session_start(); 
                }
                $_SESSION['form_error'] = 'Cannot delete bed. Bed is currently occupied.';
                $base = (defined('BASE_URL') ? BASE_URL : '');
                if ($roomId > 0) {
                    header('Location: ' . $base . '/rooms/view?id=' . urlencode((string)$roomId));
                } else {
                    header('Location: ' . $base . '/beds');
                }
                exit;
            }
            
            $stmt = $db->prepare('DELETE FROM beds WHERE id = :id');
            $stmt->execute([':id' => $bedId]);
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Bed deleted successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            if ($roomId > 0) {
                header('Location: ' . $base . '/rooms/view?id=' . urlencode((string)$roomId));
            } else {
                header('Location: ' . $base . '/beds');
            }
            exit;
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Could not delete bed.'; 
        }
    }

    public function delete(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid id'; 
            return; 
        }
        try {
            $model = new Room();
            $stmt = $model->db()->prepare('DELETE FROM rooms WHERE id = :id');
            $stmt->execute([':id' => $id]);
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/rooms');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Could not delete room.'; 
        }
    }

    public function deactivate(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid id'; 
            return; 
        }
        try {
            $model = new Room();
            $stmt = $model->db()->prepare('UPDATE rooms SET status = 0 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Room/Ward deactivated.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/rooms');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Could not deactivate room.'; 
        }
    }

    public function restore(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid id'; 
            return; 
        }
        try {
            $model = new Room();
            $stmt = $model->db()->prepare('UPDATE rooms SET status = 1 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Room/Ward restored.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/rooms');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Could not restore room.'; 
        }
    }

    protected function validateRoom(array $data, ?int $excludeId = null): array
    {
        $errors = [];
        // name required
        if (trim((string)($data['name'] ?? '')) === '') {
            $errors['name'] = 'Name is required.';
        } elseif (mb_strlen($data['name']) > 200) {
            $errors['name'] = 'Name is too long (max 200 chars).';
        }

        // code optional but max length
        if (isset($data['code']) && $data['code'] !== '' && mb_strlen($data['code']) > 50) {
            $errors['code'] = 'Code is too long (max 50 chars).';
        }

        // capacity validation
        if (isset($data['capacity']) && $data['capacity'] < 1) {
            $errors['capacity'] = 'Capacity must be at least 1.';
        }

        // charge validation
        if (isset($data['charge_per_day']) && $data['charge_per_day'] < 0) {
            $errors['charge_per_day'] = 'Charge per day cannot be negative.';
        }

        // room_type validation
        $validTypes = ['General', 'Private', 'Semi-Private', 'ICU', 'CCU', 'Emergency', 'Operation Theatre', 'Other'];
        if (isset($data['room_type']) && !in_array($data['room_type'], $validTypes, true)) {
            $errors['room_type'] = 'Invalid room type.';
        }

        // duplicate name/code checks
        try {
            $model = new Room();
            $db = $model->db();
            if (!empty($data['name'])) {
                $sql = 'SELECT id FROM rooms WHERE name = :name LIMIT 1';
                $stmt = $db->prepare($sql);
                $stmt->execute([':name' => $data['name']]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($row && (!isset($row['id']) || (int)$row['id'] !== (int)$excludeId)) {
                    $errors['name'] = 'Room name already exists.';
                }
            }
            if (!empty($data['code'])) {
                $sql = 'SELECT id FROM rooms WHERE code = :code LIMIT 1';
                $stmt = $db->prepare($sql);
                $stmt->execute([':code' => $data['code']]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($row && (!isset($row['id']) || (int)$row['id'] !== (int)$excludeId)) {
                    $errors['code'] = 'Room code already exists.';
                }
            }
        } catch (Throwable $e) {
            // ignore db check errors
        }

        return $errors;
    }
}

