<?php
declare(strict_types=1);

class ServicesController extends Controller
{
    public function index(): void
    {
        // list services (search, filter, pagination)
        $q = trim((string)($_GET['q'] ?? ''));
        $status = isset($_GET['status']) && $_GET['status'] !== '' ? (int)$_GET['status'] : null;
        $departmentId = isset($_GET['department_id']) && $_GET['department_id'] !== '' ? (int)$_GET['department_id'] : null;
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $services = [];
        $total = 0;
        $departments = [];
        try {
            $model = new Service();
            $services = $model->list(['q' => $q, 'status' => $status, 'department_id' => $departmentId, 'limit' => $perPage, 'offset' => $offset]);
            $total = $model->count(['q' => $q, 'status' => $status, 'department_id' => $departmentId]);
            
            $deptModel = new Department();
            $deptList = $deptModel->list(['status' => 1, 'limit' => 1000]);
            $departments = $deptList;
        } catch (Throwable $e) {
            $services = [];
            $total = 0;
            $departments = [];
        }

        $this->view('services/list', [
            'title' => 'Services',
            'services' => $services,
            'departments' => $departments,
            'q' => $q,
            'status' => $status,
            'department_id' => $departmentId,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
        ]);
    }

    public function create(): void
    {
        // show add form
        $departments = [];
        try {
            $deptModel = new Department();
            $departments = $deptModel->list(['status' => 1, 'limit' => 1000]);
        } catch (Throwable $e) {
            $departments = [];
        }
        $this->view('services/add', ['title' => 'Add Service', 'departments' => $departments]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        if (session_status() !== PHP_SESSION_ACTIVE) { 
            session_start(); 
        }
        unset($_SESSION['form_error'], $_SESSION['old']);

        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? '') ?: null,
            'department_id' => isset($_POST['department_id']) && (int)$_POST['department_id'] > 0 ? (int)$_POST['department_id'] : null,
            'description' => trim($_POST['description'] ?? '') ?: null,
            'price' => isset($_POST['price']) ? (float)$_POST['price'] : 0.00,
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];

        // validate input
        $errors = $this->validateService($data);
        if (!empty($errors)) {
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/services/add');
            exit;
        }

        try {
            $model = new Service();
            $id = $model->create($data);
            $_SESSION['flash_success'] = 'Service created successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/services/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            $_SESSION['old'] = $data;
            $msg = 'Could not save service.';
            if ($e instanceof PDOException) {
                $info = $e->errorInfo ?? null;
                if (is_array($info) && isset($info[1]) && intval($info[1]) === 1062) {
                    $msg = 'Service code already exists.';
                }
            }
            $_SESSION['form_error'] = $msg;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/services/add');
            exit;
        }
    }

    public function show(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid service id'; 
            return; 
        }
        try {
            $model = new Service();
            $service = $model->find($id);
            if (!$service) { 
                http_response_code(404); 
                echo 'Service not found'; 
                return; 
            }
            $this->view('services/view', [
                'title' => 'Service Details',
                'service' => $service,
            ]);
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Error loading service'; 
        }
    }

    public function edit(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid service id'; 
            return; 
        }
        try {
            $model = new Service();
            $service = $model->find($id);
            if (!$service) { 
                http_response_code(404); 
                echo 'Service not found'; 
                return; 
            }
            $deptModel = new Department();
            $departments = $deptModel->list(['status' => 1, 'limit' => 1000]);
            $this->view('services/edit', [
                'title' => 'Edit Service',
                'service' => $service,
                'departments' => $departments,
            ]);
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Error loading service for edit'; 
        }
    }

    public function update(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid service id'; 
            return; 
        }
        
        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? '') ?: null,
            'department_id' => isset($_POST['department_id']) && (int)$_POST['department_id'] > 0 ? (int)$_POST['department_id'] : null,
            'description' => trim($_POST['description'] ?? '') ?: null,
            'price' => isset($_POST['price']) ? (float)$_POST['price'] : 0.00,
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];
        
        // validate input (exclude current id when checking duplicates)
        $errors = $this->validateService($data, $id);
        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/services/edit?id=' . urlencode((string)$id));
            exit;
        }
        
        try {
            $model = new Service();
            $result = $model->update($id, $data);
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Service updated successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/services');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500);
            error_log('Service update error: ' . $e->getMessage());
            
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['old'] = $data;
            $_SESSION['form_error'] = 'Could not update service: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/services/edit?id=' . urlencode((string)$id));
            exit;
        }
    }

    public function delete(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid id'; 
            return; 
        }
        try {
            $model = new Service();
            $stmt = $model->db()->prepare('DELETE FROM services WHERE id = :id');
            $stmt->execute([':id' => $id]);
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/services');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Could not delete service.'; 
        }
    }

    public function deactivate(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid id'; 
            return; 
        }
        try {
            $model = new Service();
            $stmt = $model->db()->prepare('UPDATE services SET status = 0 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Service deactivated.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/services');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Could not deactivate service.'; 
        }
    }

    public function restore(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') { 
            http_response_code(405); 
            echo 'Method Not Allowed'; 
            return; 
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) { 
            http_response_code(400); 
            echo 'Invalid id'; 
            return; 
        }
        try {
            $model = new Service();
            $stmt = $model->db()->prepare('UPDATE services SET status = 1 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { 
                session_start(); 
            }
            $_SESSION['flash_success'] = 'Service restored.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/services');
            exit;
        } catch (Throwable $e) { 
            http_response_code(500); 
            echo 'Could not restore service.'; 
        }
    }

    /**
     * Validate service input. Returns array of field => message.
     * If $excludeId is provided it will be ignored when checking duplicate name/code.
     */
    protected function validateService(array $data, ?int $excludeId = null): array
    {
        $errors = [];
        // name required
        if (trim((string)($data['name'] ?? '')) === '') {
            $errors['name'] = 'Name is required.';
        } elseif (mb_strlen($data['name']) > 200) {
            $errors['name'] = 'Name is too long (max 200 chars).';
        }

        // code optional but max length
        if (isset($data['code']) && $data['code'] !== '' && mb_strlen($data['code']) > 50) {
            $errors['code'] = 'Code is too long (max 50 chars).';
        }

        // price validation
        if (isset($data['price']) && $data['price'] < 0) {
            $errors['price'] = 'Price cannot be negative.';
        }

        // duplicate name/code checks
        try {
            $model = new Service();
            $db = $model->db();
            if (!empty($data['name'])) {
                $sql = 'SELECT id FROM services WHERE name = :name LIMIT 1';
                $stmt = $db->prepare($sql);
                $stmt->execute([':name' => $data['name']]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($row && (!isset($row['id']) || (int)$row['id'] !== (int)$excludeId)) {
                    $errors['name'] = 'Service name already exists.';
                }
            }
            if (!empty($data['code'])) {
                $sql = 'SELECT id FROM services WHERE code = :code LIMIT 1';
                $stmt = $db->prepare($sql);
                $stmt->execute([':code' => $data['code']]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($row && (!isset($row['id']) || (int)$row['id'] !== (int)$excludeId)) {
                    $errors['code'] = 'Service code already exists.';
                }
            }
        } catch (Throwable $e) {
            // ignore db check errors, they will be surfaced on save
        }

        return $errors;
    }
}






