<?php
declare(strict_types=1);

class SettingsController extends Controller
{
    public function index(): void
    {
        $this->view('settings/index', [
            'title' => 'Settings',
        ]);
    }

    public function departments(): void
    {
        // Instantiate Doctor model directly and fetch departments
        // Read filters, search and pagination from query string
        $q = trim((string)($_GET['q'] ?? ''));
        $status = isset($_GET['status']) && $_GET['status'] !== '' ? (int)$_GET['status'] : null;
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $departments = [];
        $total = 0;
        try {
            $model = new Department();
            $departments = $model->list(['q' => $q, 'status' => $status, 'limit' => $perPage, 'offset' => $offset]);
            $total = $model->count(['q' => $q, 'status' => $status]);
        } catch (Throwable $e) {
            $departments = [];
            $total = 0;
        }

        $this->view('departments/departments', [
            'title' => 'Departments',
            'departments' => $departments,
            'q' => $q,
            'status' => $status,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
        ]);
    }

    public function addDepartment(): void
    {
        // Provide list of doctors to select HOD
        $doctors = [];
        try {
            $docModel = new Doctor();
            $doctors = $docModel->all();
        } catch (Throwable $e) {
            $doctors = [];
        }

        $this->view('departments/add', [
            'title' => 'Add Department',
            'doctors' => $doctors,
        ]);
    }

    public function storeDepartment(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        // clear any previous form/session errors
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        unset($_SESSION['form_error'], $_SESSION['old']);

        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? ''),
            'hod_id' => (int)($_POST['hod_id'] ?? 0) ?: null,
            'description' => trim($_POST['description'] ?? ''),
            'phone' => trim($_POST['phone'] ?? ''),
            'email' => trim($_POST['email'] ?? ''),
            'floor' => trim($_POST['floor'] ?? ''),
            'opd_start' => $_POST['opd_start'] ?? null,
            'opd_end' => $_POST['opd_end'] ?? null,
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];

        if ($data['name'] === '') {
            http_response_code(422);
            echo 'Name is required.';
            return;
        }

        try {
            $model = new Department();
            $id = $model->create($data);
            // set flash message and redirect to view
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Department created successfully.';
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/departments/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            // if duplicate name or DB error, preserve old input and show friendly message on add form
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $msg = 'Could not save department.';
            if ($e instanceof PDOException) {
                $info = $e->errorInfo ?? null;
                if (is_array($info) && isset($info[1]) && intval($info[1]) === 1062) {
                    $msg = 'Department name already exists.';
                }
            }
            $_SESSION['form_error'] = $msg;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/departments/add');
            exit;
        }
    }

    public function viewDepartment(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid department id';
            return;
        }
        try {
            $model = new Department();
            $dept = $model->find($id);
            if (!$dept) {
                http_response_code(404);
                echo 'Department not found';
                return;
            }
            // fetch doctors in department
            $docModel = new Doctor();
            $stmt = $this->getDoctorsByDepartment($id, $docModel);
            $this->view('departments/view', [
                'title' => 'Department Details',
                'department' => $dept,
                'doctors' => $stmt,
            ]);
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Error loading department';
        }
    }

    protected function getDoctorsByDepartment(int $deptId, Doctor $docModel): array
    {
        // simple helper using Doctor model query
        $all = $docModel->all();
        $out = [];
        foreach ($all as $d) {
            if (isset($d['department_id']) && (int)$d['department_id'] === $deptId) {
                $out[] = $d;
            }
        }
        return $out;
    }

    public function editDepartment(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid department id';
            return;
        }
        try {
            $model = new Department();
            $dept = $model->find($id);
            if (!$dept) {
                http_response_code(404);
                echo 'Department not found';
                return;
            }
            $docModel = new Doctor();
            $doctors = $docModel->all();
            $this->view('departments/edit', [
                'title' => 'Edit Department',
                'department' => $dept,
                'doctors' => $doctors,
            ]);
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Error loading department for edit';
        }
    }

    public function updateDepartment(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid department id';
            return;
        }
        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? ''),
            'hod_id' => (int)($_POST['hod_id'] ?? 0) ?: null,
            'description' => trim($_POST['description'] ?? ''),
            'phone' => trim($_POST['phone'] ?? ''),
            'email' => trim($_POST['email'] ?? ''),
            'floor' => trim($_POST['floor'] ?? ''),
            'opd_start' => $_POST['opd_start'] ?? null,
            'opd_end' => $_POST['opd_end'] ?? null,
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];

        if ($data['name'] === '') {
            http_response_code(422);
            echo 'Name is required.';
            return;
        }

        try {
            // simple update via PDO
            $sql = "UPDATE departments SET name = :name, code = :code, hod_id = :hod_id, description = :description, phone = :phone, email = :email, floor = :floor, opd_start = :opd_start, opd_end = :opd_end, status = :status WHERE id = :id";
            $model = new Department();
            $stmt = $model->db()->prepare($sql);
            $stmt->execute([
                ':name' => $data['name'],
                ':code' => $data['code'],
                ':hod_id' => $data['hod_id'],
                ':description' => $data['description'],
                ':phone' => $data['phone'],
                ':email' => $data['email'],
                ':floor' => $data['floor'],
                ':opd_start' => $data['opd_start'],
                ':opd_end' => $data['opd_end'],
                ':status' => $data['status'],
                ':id' => $id,
            ]);
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/departments');
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not update department.';
        }
    }

    public function deleteDepartment(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid id';
            return;
        }
        try {
            $model = new Department();
            $stmt = $model->db()->prepare('DELETE FROM departments WHERE id = :id');
            $stmt->execute([':id' => $id]);
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/departments');
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not delete department.';
        }
    }

    public function deactivateDepartment(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid id';
            return;
        }
        try {
            $model = new Department();
            $stmt = $model->db()->prepare('UPDATE departments SET status = 0 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Department deactivated.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/departments');
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not deactivate department.';
        }
    }

    public function restoreDepartment(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid id';
            return;
        }
        try {
            $model = new Department();
            $stmt = $model->db()->prepare('UPDATE departments SET status = 1 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Department restored.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/departments');
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not restore department.';
        }
    }
}
