<?php
declare(strict_types=1);

class StaffController extends Controller
{
    public function index(): void
    {
        $q = trim((string)($_GET['q'] ?? ''));
        $status = isset($_GET['status']) && $_GET['status'] !== '' ? (int)$_GET['status'] : null;
        $departmentId = isset($_GET['department_id']) && (int)$_GET['department_id'] > 0 ? (int)$_GET['department_id'] : null;
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $staff = [];
        $total = 0;
        $departments = [];
        try {
            $model = new Staff();
            $staff = $model->list(['q' => $q, 'status' => $status, 'department_id' => $departmentId, 'limit' => $perPage, 'offset' => $offset]);
            $total = $model->count(['q' => $q, 'status' => $status, 'department_id' => $departmentId]);
            $departments = $model->getDepartments();
        } catch (Throwable $e) {
            $staff = [];
            $total = 0;
            $departments = [];
        }

        $this->view('staff/list', [
            'title' => 'Staff Management',
            'staff' => $staff,
            'q' => $q,
            'status' => $status,
            'department_id' => $departmentId,
            'departments' => $departments,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
        ]);
    }

    public function create(): void
    {
        $departments = [];
        try {
            $model = new Staff();
            $departments = $model->getDepartments();
            $employeeId = $model->generateEmployeeId();
        } catch (Throwable $e) {
            $departments = [];
            $employeeId = 'EMP00001';
        }

        $this->view('staff/add', [
            'title' => 'Add Staff',
            'departments' => $departments,
            'employeeId' => $employeeId ?? 'EMP00001',
        ]);
    }

    public function store(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $data = [
            'employee_id' => trim($_POST['employee_id'] ?? ''),
            'first_name' => trim($_POST['first_name'] ?? ''),
            'last_name' => trim($_POST['last_name'] ?? ''),
            'gender' => trim($_POST['gender'] ?? '') ?: null,
            'dob' => trim($_POST['dob'] ?? '') ?: null,
            'phone' => trim($_POST['phone'] ?? '') ?: null,
            'email' => trim($_POST['email'] ?? '') ?: null,
            'address' => trim($_POST['address'] ?? '') ?: null,
            'department_id' => isset($_POST['department_id']) && (int)$_POST['department_id'] > 0 ? (int)$_POST['department_id'] : null,
            'designation' => trim($_POST['designation'] ?? '') ?: null,
            'joining_date' => trim($_POST['joining_date'] ?? '') ?: null,
            'salary' => isset($_POST['salary']) ? (float)$_POST['salary'] : 0.00,
            'shift' => trim($_POST['shift'] ?? 'General'),
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];

        $errors = $this->validateStaff($data);
        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/staff/add');
            exit;
        }

        try {
            $model = new Staff();
            $id = $model->create($data);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Staff member added successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/staff');
            exit;
        } catch (Throwable $e) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_error'] = 'Could not add staff: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/staff/add');
            exit;
        }
    }

    public function show(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid id';
            return;
        }

        try {
            $model = new Staff();
            $staff = $model->find($id);
            if (!$staff) {
                http_response_code(404);
                echo 'Staff not found';
                return;
            }

            // Get attendance records
            $attendanceModel = new Attendance();
            $attendance = $attendanceModel->getStaffAttendance($id, date('Y-m-01'), date('Y-m-t'));

            $this->view('staff/view', [
                'title' => 'Staff Details',
                'staff' => $staff,
                'attendance' => $attendance,
            ]);
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Error loading staff';
        }
    }

    public function edit(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid id';
            return;
        }

        try {
            $model = new Staff();
            $staff = $model->find($id);
            if (!$staff) {
                http_response_code(404);
                echo 'Staff not found';
                return;
            }

            $departments = $model->getDepartments();
            $this->view('staff/edit', [
                'title' => 'Edit Staff',
                'staff' => $staff,
                'departments' => $departments,
            ]);
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Error loading staff';
        }
    }

    public function update(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid id';
            return;
        }

        $data = [
            'employee_id' => trim($_POST['employee_id'] ?? ''),
            'first_name' => trim($_POST['first_name'] ?? ''),
            'last_name' => trim($_POST['last_name'] ?? ''),
            'gender' => trim($_POST['gender'] ?? '') ?: null,
            'dob' => trim($_POST['dob'] ?? '') ?: null,
            'phone' => trim($_POST['phone'] ?? '') ?: null,
            'email' => trim($_POST['email'] ?? '') ?: null,
            'address' => trim($_POST['address'] ?? '') ?: null,
            'department_id' => isset($_POST['department_id']) && (int)$_POST['department_id'] > 0 ? (int)$_POST['department_id'] : null,
            'designation' => trim($_POST['designation'] ?? '') ?: null,
            'joining_date' => trim($_POST['joining_date'] ?? '') ?: null,
            'salary' => isset($_POST['salary']) ? (float)$_POST['salary'] : 0.00,
            'shift' => trim($_POST['shift'] ?? 'General'),
            'status' => isset($_POST['status']) ? (int)$_POST['status'] : 1,
        ];

        $errors = $this->validateStaff($data, $id);
        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/staff/edit?id=' . urlencode((string)$id));
            exit;
        }

        try {
            $model = new Staff();
            $model->update($id, $data);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Staff updated successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/staff');
            exit;
        } catch (Throwable $e) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_error'] = 'Could not update staff: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/staff/edit?id=' . urlencode((string)$id));
            exit;
        }
    }

    public function delete(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid id';
            return;
        }

        try {
            $model = new Staff();
            $stmt = $model->db()->prepare('DELETE FROM staff WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Staff deleted successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/staff');
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not delete staff.';
        }
    }

    public function deactivate(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid id';
            return;
        }

        try {
            $model = new Staff();
            $stmt = $model->db()->prepare('UPDATE staff SET status = 0 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Staff deactivated.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/staff');
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not deactivate staff.';
        }
    }

    public function restore(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid id';
            return;
        }

        try {
            $model = new Staff();
            $stmt = $model->db()->prepare('UPDATE staff SET status = 1 WHERE id = :id');
            $stmt->execute([':id' => $id]);
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Staff restored.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/staff');
            exit;
        } catch (Throwable $e) {
            http_response_code(500);
            echo 'Could not restore staff.';
        }
    }

    protected function validateStaff(array $data, ?int $excludeId = null): array
    {
        $errors = [];

        if (trim((string)($data['first_name'] ?? '')) === '') {
            $errors['first_name'] = 'First name is required.';
        }

        if (trim((string)($data['last_name'] ?? '')) === '') {
            $errors['last_name'] = 'Last name is required.';
        }

        if (trim((string)($data['employee_id'] ?? '')) === '') {
            $errors['employee_id'] = 'Employee ID is required.';
        } else {
            // Check duplicate employee_id
            try {
                $model = new Staff();
                $db = $model->db();
                $sql = 'SELECT id FROM staff WHERE employee_id = :emp_id LIMIT 1';
                $stmt = $db->prepare($sql);
                $stmt->execute([':emp_id' => $data['employee_id']]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($row && (!isset($row['id']) || (int)$row['id'] !== (int)$excludeId)) {
                    $errors['employee_id'] = 'Employee ID already exists.';
                }
            } catch (Throwable $e) {
                // ignore
            }
        }

        if (!empty($data['email']) && !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Invalid email format.';
        }

        return $errors;
    }
}






