<?php
declare(strict_types=1);

class Department extends Model
{
    /**
     * Fetch departments with optional search/status filter, pagination
     * returns array of rows with extra columns: total_doctors, hod_name
     */
    public function list(array $opts = []): array
    {
        $q = trim((string)($opts['q'] ?? ''));
        $status = isset($opts['status']) ? (int)$opts['status'] : null;
        $hospitalId = isset($opts['hospital_id']) ? (int)$opts['hospital_id'] : null;
        $limit = isset($opts['limit']) ? (int)$opts['limit'] : 20;
        $offset = isset($opts['offset']) ? (int)$opts['offset'] : 0;

        $where = [];
        $params = [];
        if ($hospitalId !== null) {
            $where[] = 'd.hospital_id = :hospital_id';
            $params[':hospital_id'] = $hospitalId;
        }
        if ($q !== '') {
            $where[] = '(d.name LIKE :q OR d.code LIKE :q OR d.description LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($status !== null) {
            $where[] = 'd.status = :status';
            $params[':status'] = $status;
        }
        $whereSql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

        // Build SQL - subquery doesn't need hospital_id param binding, it's scoped to main query
        $sql = "SELECT d.*, 
                   (SELECT COUNT(*) FROM doctors doc WHERE doc.department_id = d.id" . ($hospitalId !== null ? " AND doc.hospital_id = d.hospital_id" : "") . ") AS total_doctors,
                   (SELECT CONCAT(first_name, ' ', last_name) FROM doctors h WHERE h.id = d.hod_id LIMIT 1) AS hod_name
                FROM departments d
                $whereSql
                ORDER BY d.name ASC
                LIMIT :limit OFFSET :offset";

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    public function count(array $opts = []): int
    {
        $q = trim((string)($opts['q'] ?? ''));
        $status = isset($opts['status']) ? (int)$opts['status'] : null;
        $hospitalId = isset($opts['hospital_id']) ? (int)$opts['hospital_id'] : null;

        $where = [];
        $params = [];
        if ($hospitalId !== null) {
            $where[] = 'hospital_id = :hospital_id';
            $params[':hospital_id'] = $hospitalId;
        }
        if ($q !== '') {
            $where[] = '(name LIKE :q OR code LIKE :q OR description LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($status !== null) {
            $where[] = 'status = :status';
            $params[':status'] = $status;
        }
        $whereSql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

        $sql = "SELECT COUNT(*) FROM departments $whereSql";
        $stmt = $this->db()->prepare($sql);
        foreach ($params as $k => $v) { $stmt->bindValue($k, $v); }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function find(int $id, ?int $hospitalId = null): ?array
    {
        $sql = "SELECT d.*, 
                   (SELECT COUNT(*) FROM doctors doc WHERE doc.department_id = d.id" . ($hospitalId !== null ? " AND doc.hospital_id = :hospital_id" : "") . ") AS total_doctors,
                   (SELECT CONCAT(first_name, ' ', last_name) FROM doctors h WHERE h.id = d.hod_id LIMIT 1) AS hod_name
                FROM departments d
                WHERE d.id = :id";
        if ($hospitalId !== null) {
            $sql .= " AND d.hospital_id = :hospital_id";
        }
        $sql .= " LIMIT 1";
        $stmt = $this->db()->prepare($sql);
        $params = [':id' => $id];
        if ($hospitalId !== null) {
            $params[':hospital_id'] = $hospitalId;
        }
        $stmt->execute($params);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $sql = "INSERT INTO departments (hospital_id, name, code, hod_id, description, phone, email, floor, opd_start, opd_end, status, created_at)
                VALUES (:hospital_id, :name, :code, :hod_id, :description, :phone, :email, :floor, :opd_start, :opd_end, :status, NOW())";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':hospital_id' => $data['hospital_id'] ?? null,
            ':name' => $data['name'] ?? null,
            ':code' => $data['code'] ?? null,
            ':hod_id' => $data['hod_id'] ?? null,
            ':description' => $data['description'] ?? null,
            ':phone' => $data['phone'] ?? null,
            ':email' => $data['email'] ?? null,
            ':floor' => $data['floor'] ?? null,
            ':opd_start' => $data['opd_start'] ?? null,
            ':opd_end' => $data['opd_end'] ?? null,
            ':status' => isset($data['status']) ? (int)$data['status'] : 1,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $sql = "UPDATE departments SET 
                name = :name, 
                code = :code, 
                hod_id = :hod_id, 
                description = :description, 
                phone = :phone, 
                email = :email, 
                floor = :floor, 
                opd_start = :opd_start, 
                opd_end = :opd_end, 
                status = :status 
                WHERE id = :id";
        
        $stmt = $this->db()->prepare($sql);
        return $stmt->execute([
            ':name' => $data['name'] ?? null,
            ':code' => $data['code'] ?? null,
            ':hod_id' => $data['hod_id'] ?? null,
            ':description' => $data['description'] ?? null,
            ':phone' => $data['phone'] ?? null,
            ':email' => $data['email'] ?? null,
            ':floor' => $data['floor'] ?? null,
            ':opd_start' => $data['opd_start'] ?? null,
            ':opd_end' => $data['opd_end'] ?? null,
            ':status' => isset($data['status']) ? (int)$data['status'] : 1,
            ':id' => $id,
        ]);
    }

    public function all(?int $hospitalId = null): array
    {
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare("SELECT id, name, code FROM departments WHERE status = 1 AND hospital_id = :hospital_id ORDER BY name");
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query("SELECT id, name, code FROM departments WHERE status = 1 ORDER BY name");
        }
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
