<?php
declare(strict_types=1);

class Admission extends Model
{
    public function all(): array
    {
        $sql = "SELECT a.id, a.admitted_at, a.discharged_at, a.diagnosis,
                       p.first_name AS p_first, p.last_name AS p_last,
                       d.first_name AS d_first, d.last_name AS d_last,
                       b.ward, b.number
                FROM admissions a
                INNER JOIN patients p ON p.id = a.patient_id
                LEFT JOIN doctors d ON d.id = a.doctor_id
                LEFT JOIN beds b ON b.id = a.bed_id
                ORDER BY a.admitted_at DESC";
        $stmt = $this->db()->query($sql);
        return $stmt->fetchAll();
    }

    public function list(array $filters = []): array
    {
        $q = $filters['q'] ?? '';
        $doctorId = $filters['doctor_id'] ?? null;
        $status = $filters['status'] ?? null;
        $hospitalId = $filters['hospital_id'] ?? null;
        $limit = $filters['limit'] ?? 20;
        $offset = $filters['offset'] ?? 0;

        $where = [];
        $params = [];

        $sql = "SELECT a.id, a.admitted_at, a.discharged_at, a.diagnosis,
                       p.id AS patient_id, p.first_name AS p_first, p.last_name AS p_last, p.phone AS patient_phone, p.gender AS patient_gender,
                       d.id AS doctor_id, d.first_name AS d_first, d.last_name AS d_last, d.specialty AS doctor_specialty,
                       b.ward, b.number AS bed_number, b.type AS bed_type,
                       dept.name AS department_name
                FROM admissions a
                INNER JOIN patients p ON p.id = a.patient_id
                LEFT JOIN doctors d ON d.id = a.doctor_id
                LEFT JOIN departments dept ON dept.id = d.department_id
                LEFT JOIN beds b ON b.id = a.bed_id
                WHERE 1=1";

        // Filter by hospital_id if provided
        if ($hospitalId !== null) {
            $where[] = 'a.hospital_id = :hospital_id';
            $params[':hospital_id'] = $hospitalId;
        }

        if ($q !== '') {
            $where[] = '(p.first_name LIKE :q OR p.last_name LIKE :q OR p.phone LIKE :q OR d.first_name LIKE :q OR d.last_name LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($doctorId !== null) {
            $where[] = 'a.doctor_id = :doctor_id';
            $params[':doctor_id'] = $doctorId;
        }
        if ($status !== null && $status !== '') {
            if ($status === 'Active') {
                $where[] = 'a.discharged_at IS NULL';
            } elseif ($status === 'Discharged') {
                $where[] = 'a.discharged_at IS NOT NULL';
            }
        }

        if (!empty($where)) {
            $sql .= ' AND ' . implode(' AND ', $where);
        }

        $sql .= " ORDER BY a.admitted_at DESC LIMIT :limit OFFSET :offset";

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function count(array $filters = []): int
    {
        $q = $filters['q'] ?? '';
        $doctorId = $filters['doctor_id'] ?? null;
        $status = $filters['status'] ?? null;
        $hospitalId = $filters['hospital_id'] ?? null;

        $where = [];
        $params = [];

        $sql = "SELECT COUNT(*) FROM admissions a
                INNER JOIN patients p ON p.id = a.patient_id
                LEFT JOIN doctors d ON d.id = a.doctor_id
                WHERE 1=1";

        // Filter by hospital_id if provided
        if ($hospitalId !== null) {
            $where[] = 'a.hospital_id = :hospital_id';
            $params[':hospital_id'] = $hospitalId;
        }

        if ($q !== '') {
            $where[] = '(p.first_name LIKE :q OR p.last_name LIKE :q OR p.phone LIKE :q OR d.first_name LIKE :q OR d.last_name LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($doctorId !== null) {
            $where[] = 'a.doctor_id = :doctor_id';
            $params[':doctor_id'] = $doctorId;
        }
        if ($status !== null && $status !== '') {
            if ($status === 'Active') {
                $where[] = 'a.discharged_at IS NULL';
            } elseif ($status === 'Discharged') {
                $where[] = 'a.discharged_at IS NOT NULL';
            }
        }

        if (!empty($where)) {
            $sql .= ' AND ' . implode(' AND ', $where);
        }

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function find(int $id, ?int $hospitalId = null): ?array
    {
        $sql = "SELECT a.*, 
                       p.id AS patient_id, p.first_name AS p_first, p.last_name AS p_last, 
                       p.phone AS patient_phone, p.email AS patient_email, p.gender AS patient_gender,
                       p.dob AS patient_dob, p.address AS patient_address,
                       d.id AS doctor_id, d.first_name AS d_first, d.last_name AS d_last,
                       d.specialty AS doctor_specialty, d.phone AS doctor_phone, d.email AS doctor_email,
                       b.ward, b.number AS bed_number, b.type AS bed_type,
                       dept.name AS department_name, dept.code AS department_code
                FROM admissions a
                INNER JOIN patients p ON p.id = a.patient_id
                LEFT JOIN doctors d ON d.id = a.doctor_id
                LEFT JOIN departments dept ON dept.id = d.department_id
                LEFT JOIN beds b ON b.id = a.bed_id
                WHERE a.id = :id";
        if ($hospitalId !== null) {
            $sql .= " AND a.hospital_id = :hospital_id";
        }
        $stmt = $this->db()->prepare($sql);
        $params = [':id' => $id];
        if ($hospitalId !== null) {
            $params[':hospital_id'] = $hospitalId;
        }
        $stmt->execute($params);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $pdo = $this->db();
        $pdo->beginTransaction();
        try {
            $stmt = $pdo->prepare("INSERT INTO admissions (hospital_id, patient_id, doctor_id, bed_id, admitted_at, diagnosis) VALUES (:hid, :pid, :did, :bid, NOW(), :diag)");
            $stmt->execute([
                ':hid' => $data['hospital_id'] ?? null,
                ':pid' => $data['patient_id'],
                ':did' => $data['doctor_id'] ?: null,
                ':bid' => $data['bed_id'] ?: null,
                ':diag' => $data['diagnosis'] ?? null,
            ]);
            $id = (int)$pdo->lastInsertId();
            if (!empty($data['bed_id'])) {
                $upd = $pdo->prepare('UPDATE beds SET is_available = 0 WHERE id = :id');
                $upd->execute([':id' => $data['bed_id']]);
            }
            $pdo->commit();
            return $id;
        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }

    public function discharge(int $id): void
    {
        $pdo = $this->db();
        $pdo->beginTransaction();
        try {
            // Lock the admission row and fetch details along with patient for summary
            $stmt = $pdo->prepare('SELECT a.id, a.patient_id, a.bed_id, a.admitted_at, a.diagnosis, a.discharged_at, p.first_name, p.last_name FROM admissions a INNER JOIN patients p ON p.id = a.patient_id WHERE a.id = :id FOR UPDATE');
            $stmt->execute([':id' => $id]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$row) {
                throw new RuntimeException('Admission not found');
            }
            $bedId = $row['bed_id'] ?? null;

            // Mark discharged
            $upd = $pdo->prepare('UPDATE admissions SET discharged_at = NOW() WHERE id = :id AND discharged_at IS NULL');
            $upd->execute([':id' => $id]);

            // Ensure discharge_summaries table exists
            $pdo->exec("CREATE TABLE IF NOT EXISTS discharge_summaries (
                id INT AUTO_INCREMENT PRIMARY KEY,
                patient_id INT NULL,
                patient_name VARCHAR(255) NULL,
                admission_date DATETIME NULL,
                discharge_date DATETIME NULL,
                diagnosis TEXT NULL,
                treatment TEXT NULL,
                medications TEXT NULL,
                follow_up TEXT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX (patient_id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

            // Create discharge summary record
            $ins = $pdo->prepare("INSERT INTO discharge_summaries (patient_id, patient_name, admission_date, discharge_date, diagnosis, treatment, medications, follow_up) VALUES (:patient_id, :patient_name, :admission_date, NOW(), :diagnosis, NULL, NULL, NULL)");
            $patientName = trim(($row['first_name'] ?? '') . ' ' . ($row['last_name'] ?? '')) ?: null;
            $ins->execute([
                ':patient_id' => (int)$row['patient_id'],
                ':patient_name' => $patientName,
                ':admission_date' => $row['admitted_at'] ?? null,
                ':diagnosis' => $row['diagnosis'] ?? null,
            ]);
            if ($bedId) {
                $free = $pdo->prepare('UPDATE beds SET is_available = 1 WHERE id = :id');
                $free->execute([':id' => (int)$bedId]);
            }
            $pdo->commit();
        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }

    public function patients(): array
    {
        $stmt = $this->db()->query("SELECT id, first_name, last_name FROM patients ORDER BY last_name, first_name");
        return $stmt->fetchAll();
    }

    public function doctors(): array
    {
        $stmt = $this->db()->query("SELECT id, first_name, last_name, specialty, department_id FROM doctors ORDER BY last_name, first_name");
        return $stmt->fetchAll();
    }

    public function availableBeds(): array
    {
        $stmt = $this->db()->query("SELECT id, ward, number, type FROM beds WHERE is_available = 1 ORDER BY ward, number");
        return $stmt->fetchAll();
    }

    public function countTodayAdmissions(?int $hospitalId = null): int
    {
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare('SELECT COUNT(*) FROM admissions WHERE hospital_id = :hospital_id AND DATE(admitted_at)=CURDATE()');
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query('SELECT COUNT(*) FROM admissions WHERE DATE(admitted_at)=CURDATE()');
        }
        return (int)$stmt->fetchColumn();
    }

    public function countTodayDischarges(?int $hospitalId = null): int
    {
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare('SELECT COUNT(*) FROM admissions WHERE hospital_id = :hospital_id AND DATE(discharged_at)=CURDATE()');
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query('SELECT COUNT(*) FROM admissions WHERE DATE(discharged_at)=CURDATE()');
        }
        return (int)$stmt->fetchColumn();
    }
}
