<?php
declare(strict_types=1);

class Appointment extends Model
{
    public function all(?int $hospitalId = null): array
    {
        $sql = "SELECT a.id, a.scheduled_at, a.status, p.first_name AS p_first, p.last_name AS p_last, d.first_name AS d_first, d.last_name AS d_last
                FROM appointments a
                INNER JOIN patients p ON p.id = a.patient_id
                INNER JOIN doctors d ON d.id = a.doctor_id";
        if ($hospitalId !== null) {
            $sql .= " WHERE a.hospital_id = :hospital_id";
        }
        $sql .= " ORDER BY a.scheduled_at DESC";
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare($sql);
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query($sql);
        }
        return $stmt->fetchAll();
    }

    public function find(int $id, ?int $hospitalId = null): ?array
    {
        $sql = "SELECT a.*, p.first_name AS p_first, p.last_name AS p_last, d.first_name AS d_first, d.last_name AS d_last
                FROM appointments a
                INNER JOIN patients p ON p.id = a.patient_id
                INNER JOIN doctors d ON d.id = a.doctor_id
                WHERE a.id = :id";
        if ($hospitalId !== null) {
            $sql .= " AND a.hospital_id = :hospital_id";
        }
        $stmt = $this->db()->prepare($sql);
        $params = [':id' => $id];
        if ($hospitalId !== null) {
            $params[':hospital_id'] = $hospitalId;
        }
        $stmt->execute($params);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $sql = "INSERT INTO appointments (hospital_id, patient_id, doctor_id, scheduled_at, status, notes) VALUES (:hospital_id, :patient_id, :doctor_id, :scheduled_at, :status, :notes)";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':hospital_id' => $data['hospital_id'] ?? null,
            ':patient_id' => $data['patient_id'],
            ':doctor_id' => $data['doctor_id'],
            ':scheduled_at' => $data['scheduled_at'],
            ':status' => $data['status'] ?? 'Booked',
            ':notes' => $data['notes'] ?? null,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function patients(?int $hospitalId = null): array
    {
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare("SELECT id, first_name, last_name FROM patients WHERE hospital_id = :hospital_id ORDER BY last_name, first_name");
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query("SELECT id, first_name, last_name FROM patients ORDER BY last_name, first_name");
        }
        return $stmt->fetchAll();
    }

    public function doctors(?int $hospitalId = null): array
    {
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare("SELECT id, first_name, last_name, specialty, department_id FROM doctors WHERE hospital_id = :hospital_id ORDER BY last_name, first_name");
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query("SELECT id, first_name, last_name, specialty, department_id FROM doctors ORDER BY last_name, first_name");
        }
        return $stmt->fetchAll();
    }

    public function countAll(?int $hospitalId = null): int
    {
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare('SELECT COUNT(*) FROM appointments WHERE hospital_id = :hospital_id');
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query('SELECT COUNT(*) FROM appointments');
        }
        return (int)$stmt->fetchColumn();
    }

    public function countToday(?int $hospitalId = null): int
    {
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare("SELECT COUNT(*) FROM appointments WHERE hospital_id = :hospital_id AND DATE(scheduled_at) = CURDATE()");
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query("SELECT COUNT(*) FROM appointments WHERE DATE(scheduled_at) = CURDATE()");
        }
        return (int)$stmt->fetchColumn();
    }

    public function recent(int $limit = 10, ?int $hospitalId = null): array
    {
        $sql = "SELECT a.id, a.scheduled_at, a.status,
                       CONCAT(p.first_name,' ',p.last_name) AS patient,
                       CONCAT(d.first_name,' ',d.last_name) AS doctor
                FROM appointments a
                INNER JOIN patients p ON p.id = a.patient_id
                INNER JOIN doctors d ON d.id = a.doctor_id";
        if ($hospitalId !== null) {
            $sql .= " WHERE a.hospital_id = :hospital_id";
        }
        $sql .= " ORDER BY a.scheduled_at DESC LIMIT :lim";
        $stmt = $this->db()->prepare($sql);
        if ($hospitalId !== null) {
            $stmt->bindValue(':hospital_id', $hospitalId, PDO::PARAM_INT);
        }
        $stmt->bindValue(':lim', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }
}
