<?php
declare(strict_types=1);

class Attendance extends Model
{
    public function markAttendance(int $staffId, string $date, ?string $checkIn = null, ?string $checkOut = null, string $status = 'Present', ?string $remarks = null): int
    {
        $sql = "INSERT INTO attendance (staff_id, attendance_date, check_in, check_out, status, remarks)
                VALUES (:staff_id, :date, :check_in, :check_out, :status, :remarks)
                ON DUPLICATE KEY UPDATE
                check_in = COALESCE(:check_in, check_in),
                check_out = COALESCE(:check_out, check_out),
                status = :status,
                remarks = :remarks";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':staff_id' => $staffId,
            ':date' => $date,
            ':check_in' => $checkIn,
            ':check_out' => $checkOut,
            ':status' => $status,
            ':remarks' => $remarks,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function getAttendanceByDate(string $date): array
    {
        $sql = "SELECT a.*, s.employee_id, s.first_name, s.last_name, s.designation, d.name AS department_name
                FROM attendance a
                INNER JOIN staff s ON s.id = a.staff_id
                LEFT JOIN departments d ON d.id = s.department_id
                WHERE a.attendance_date = :date
                ORDER BY s.last_name, s.first_name";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([':date' => $date]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getStaffAttendance(int $staffId, ?string $startDate = null, ?string $endDate = null): array
    {
        $sql = "SELECT a.* FROM attendance a WHERE a.staff_id = :staff_id";
        $params = [':staff_id' => $staffId];

        if ($startDate) {
            $sql .= " AND a.attendance_date >= :start_date";
            $params[':start_date'] = $startDate;
        }

        if ($endDate) {
            $sql .= " AND a.attendance_date <= :end_date";
            $params[':end_date'] = $endDate;
        }

        $sql .= " ORDER BY a.attendance_date DESC";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getMonthlyReport(string $month, ?int $departmentId = null): array
    {
        $sql = "SELECT 
                    s.id AS staff_id,
                    s.employee_id,
                    s.first_name,
                    s.last_name,
                    s.designation,
                    d.name AS department_name,
                    COUNT(a.id) AS total_days,
                    SUM(CASE WHEN a.status = 'Present' THEN 1 ELSE 0 END) AS present_days,
                    SUM(CASE WHEN a.status = 'Absent' THEN 1 ELSE 0 END) AS absent_days,
                    SUM(CASE WHEN a.status = 'Half Day' THEN 1 ELSE 0 END) AS half_days,
                    SUM(CASE WHEN a.status = 'Leave' THEN 1 ELSE 0 END) AS leave_days
                FROM staff s
                LEFT JOIN departments d ON d.id = s.department_id
                LEFT JOIN attendance a ON a.staff_id = s.id AND DATE_FORMAT(a.attendance_date, '%Y-%m') = :month
                WHERE s.status = 1";
        
        $params = [':month' => $month];

        if ($departmentId !== null) {
            $sql .= " AND s.department_id = :dept";
            $params[':dept'] = $departmentId;
        }

        $sql .= " GROUP BY s.id ORDER BY s.last_name, s.first_name";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function find(int $id): ?array
    {
        $sql = "SELECT a.*, s.employee_id, s.first_name, s.last_name, s.designation
                FROM attendance a
                INNER JOIN staff s ON s.id = a.staff_id
                WHERE a.id = :id";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([':id' => $id]);
        return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
    }

    public function update(int $id, array $data): bool
    {
        $sql = "UPDATE attendance SET
                check_in = :check_in,
                check_out = :check_out,
                status = :status,
                remarks = :remarks
                WHERE id = :id";
        $stmt = $this->db()->prepare($sql);
        return $stmt->execute([
            ':check_in' => $data['check_in'] ?? null,
            ':check_out' => $data['check_out'] ?? null,
            ':status' => $data['status'] ?? 'Present',
            ':remarks' => $data['remarks'] ?? null,
            ':id' => $id,
        ]);
    }
}






