<?php
declare(strict_types=1);

class Invoice extends Model
{
    public function all(): array
    {
        $sql = "SELECT i.id, i.invoice_date, i.total, i.status, p.first_name AS p_first, p.last_name AS p_last
                FROM invoices i
                INNER JOIN patients p ON p.id = i.patient_id
                ORDER BY i.id DESC";
        $stmt = $this->db()->query($sql);
        return $stmt->fetchAll();
    }

    public function patients(): array
    {
        $stmt = $this->db()->query("SELECT id, first_name, last_name FROM patients ORDER BY last_name, first_name");
        return $stmt->fetchAll();
    }

    public function find(int $id): ?array
    {
        $pdo = $this->db();
        $stmt = $pdo->prepare("SELECT i.*, p.first_name AS p_first, p.last_name AS p_last FROM invoices i INNER JOIN patients p ON p.id = i.patient_id WHERE i.id = :id");
        $stmt->execute([':id' => $id]);
        $invoice = $stmt->fetch();
        if (!$invoice) return null;

        $items = $pdo->prepare("SELECT * FROM invoice_items WHERE invoice_id = :id ORDER BY id");
        $items->execute([':id' => $id]);
        $invoice['items'] = $items->fetchAll();

        $payments = $pdo->prepare("SELECT * FROM payments WHERE invoice_id = :id ORDER BY paid_at DESC, id DESC");
        $payments->execute([':id' => $id]);
        $invoice['payments'] = $payments->fetchAll();

        return $invoice;
    }

    public function create(int $patientId, array $items): int
    {
        $pdo = $this->db();
        $pdo->beginTransaction();
        try {
            $total = 0.0;
            foreach ($items as $it) {
                $qty = (float)($it['qty'] ?? 0);
                $price = (float)($it['unit_price'] ?? 0);
                $amount = $qty * $price;
                $total += $amount;
            }
            $stmt = $pdo->prepare("INSERT INTO invoices (patient_id, invoice_date, total, status) VALUES (:pid, NOW(), :total, 'Unpaid')");
            $stmt->execute([':pid' => $patientId, ':total' => $total]);
            $invoiceId = (int)$pdo->lastInsertId();

            $itemStmt = $pdo->prepare("INSERT INTO invoice_items (invoice_id, description, qty, unit_price, amount) VALUES (:iid, :desc, :qty, :price, :amount)");
            foreach ($items as $it) {
                $desc = (string)($it['description'] ?? 'Item');
                $qty = (float)($it['qty'] ?? 0);
                $price = (float)($it['unit_price'] ?? 0);
                $amount = $qty * $price;
                if ($qty <= 0 || $price < 0) { continue; }
                $itemStmt->execute([
                    ':iid' => $invoiceId,
                    ':desc' => $desc,
                    ':qty' => $qty,
                    ':price' => $price,
                    ':amount' => $amount,
                ]);
            }

            $pdo->commit();
            return $invoiceId;
        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }

    public function addPayment(int $invoiceId, float $amount, ?string $method = null): void
    {
        $pdo = $this->db();
        $stmt = $pdo->prepare("INSERT INTO payments (invoice_id, paid_at, amount, method) VALUES (:iid, NOW(), :amt, :method)");
        $stmt->execute([':iid' => $invoiceId, ':amt' => $amount, ':method' => $method]);

        // Update invoice status if fully paid
        $tot = (float)$pdo->query("SELECT total FROM invoices WHERE id = " . (int)$invoiceId)->fetchColumn();
        $paid = (float)$pdo->query("SELECT COALESCE(SUM(amount),0) FROM payments WHERE invoice_id = " . (int)$invoiceId)->fetchColumn();
        $status = $paid >= $tot ? 'Paid' : 'Unpaid';
        $upd = $pdo->prepare("UPDATE invoices SET status = :st WHERE id = :id");
        $upd->execute([':st' => $status, ':id' => $invoiceId]);
    }
}
