<?php
declare(strict_types=1);

class Charge extends Model
{
    /**
     * Fetch charges with optional search/status filter, pagination
     * returns array of rows with extra columns: department_name
     */
    public function list(array $opts = []): array
    {
        $q = trim((string)($opts['q'] ?? ''));
        $status = isset($opts['status']) ? (int)$opts['status'] : null;
        $chargeType = isset($opts['charge_type']) && $opts['charge_type'] !== '' ? (string)$opts['charge_type'] : null;
        $departmentId = isset($opts['department_id']) ? (int)$opts['department_id'] : null;
        $limit = isset($opts['limit']) ? (int)$opts['limit'] : 20;
        $offset = isset($opts['offset']) ? (int)$opts['offset'] : 0;

        $where = [];
        $params = [];
        if ($q !== '') {
            $where[] = '(c.name LIKE :q OR c.code LIKE :q OR c.description LIKE :q OR c.category LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($status !== null) {
            $where[] = 'c.status = :status';
            $params[':status'] = $status;
        }
        if ($chargeType !== null) {
            $where[] = 'c.charge_type = :charge_type';
            $params[':charge_type'] = $chargeType;
        }
        if ($departmentId !== null) {
            $where[] = 'c.department_id = :department_id';
            $params[':department_id'] = $departmentId;
        }
        $whereSql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

        $sql = "SELECT c.*, 
                   d.name AS department_name
                FROM charges c
                LEFT JOIN departments d ON d.id = c.department_id
                $whereSql
                ORDER BY c.charge_type ASC, c.name ASC
                LIMIT :limit OFFSET :offset";

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    public function count(array $opts = []): int
    {
        $q = trim((string)($opts['q'] ?? ''));
        $status = isset($opts['status']) ? (int)$opts['status'] : null;
        $chargeType = isset($opts['charge_type']) && $opts['charge_type'] !== '' ? (string)$opts['charge_type'] : null;
        $departmentId = isset($opts['department_id']) ? (int)$opts['department_id'] : null;

        $where = [];
        $params = [];
        if ($q !== '') {
            $where[] = '(name LIKE :q OR code LIKE :q OR description LIKE :q OR category LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($status !== null) {
            $where[] = 'status = :status';
            $params[':status'] = $status;
        }
        if ($chargeType !== null) {
            $where[] = 'charge_type = :charge_type';
            $params[':charge_type'] = $chargeType;
        }
        if ($departmentId !== null) {
            $where[] = 'department_id = :department_id';
            $params[':department_id'] = $departmentId;
        }
        $whereSql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

        $sql = "SELECT COUNT(*) FROM charges $whereSql";
        $stmt = $this->db()->prepare($sql);
        foreach ($params as $k => $v) { 
            $stmt->bindValue($k, $v); 
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function find(int $id): ?array
    {
        $sql = "SELECT c.*, 
                   d.name AS department_name
                FROM charges c
                LEFT JOIN departments d ON d.id = c.department_id
                WHERE c.id = :id
                LIMIT 1";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([':id' => $id]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $sql = "INSERT INTO charges (name, code, charge_type, department_id, category, description, amount, tax_percentage, status, created_at)
                VALUES (:name, :code, :charge_type, :department_id, :category, :description, :amount, :tax_percentage, :status, NOW())";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':name' => $data['name'] ?? null,
            ':code' => $data['code'] ?? null,
            ':charge_type' => $data['charge_type'] ?? 'Other',
            ':department_id' => isset($data['department_id']) && (int)$data['department_id'] > 0 ? (int)$data['department_id'] : null,
            ':category' => $data['category'] ?? null,
            ':description' => $data['description'] ?? null,
            ':amount' => isset($data['amount']) ? (float)$data['amount'] : 0.00,
            ':tax_percentage' => isset($data['tax_percentage']) ? (float)$data['tax_percentage'] : 0.00,
            ':status' => isset($data['status']) ? (int)$data['status'] : 1,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $sql = "UPDATE charges SET 
                name = :name, 
                code = :code, 
                charge_type = :charge_type, 
                department_id = :department_id, 
                category = :category, 
                description = :description, 
                amount = :amount, 
                tax_percentage = :tax_percentage, 
                status = :status 
                WHERE id = :id";
        
        $stmt = $this->db()->prepare($sql);
        return $stmt->execute([
            ':name' => $data['name'] ?? null,
            ':code' => $data['code'] ?? null,
            ':charge_type' => $data['charge_type'] ?? 'Other',
            ':department_id' => isset($data['department_id']) && (int)$data['department_id'] > 0 ? (int)$data['department_id'] : null,
            ':category' => $data['category'] ?? null,
            ':description' => $data['description'] ?? null,
            ':amount' => isset($data['amount']) ? (float)$data['amount'] : 0.00,
            ':tax_percentage' => isset($data['tax_percentage']) ? (float)$data['tax_percentage'] : 0.00,
            ':status' => isset($data['status']) ? (int)$data['status'] : 1,
            ':id' => $id,
        ]);
    }

    public function all(): array
    {
        $sql = "SELECT * FROM charges WHERE status = 1 ORDER BY charge_type ASC, name ASC";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    public function getChargeTypes(): array
    {
        return ['Consultation', 'Room', 'Operation', 'Test', 'Procedure', 'Medicine', 'Other'];
    }
}






