<?php
declare(strict_types=1);

class Dashboard extends Model
{
    /**
     * Get dashboard statistics for a hospital
     */
    public function getStats(?int $hospitalId = null): array
    {
        $patientModel = new Patient();
        $doctorModel = new Doctor();
        $apptModel = new Appointment();
        $bedModel = new Bed();

        return [
            'patients' => $this->safeCount(fn() => $patientModel->countAll($hospitalId)),
            'doctors' => $this->safeCount(fn() => $doctorModel->countAll($hospitalId)),
            'appointments' => $this->safeCount(fn() => $apptModel->countAll($hospitalId)),
            'beds_available' => $this->safeCount(fn() => $bedModel->countAvailable($hospitalId)),
        ];
    }

    /**
     * Get today's metrics
     */
    public function getTodayMetrics(?int $hospitalId = null): array
    {
        $admModel = new Admission();
        $apptModel = new Appointment();

        return [
            'appointments' => $this->safeCount(fn() => $apptModel->countToday($hospitalId)),
            'admissions' => $this->safeCount(fn() => $admModel->countTodayAdmissions($hospitalId)),
            'discharges' => $this->safeCount(fn() => $admModel->countTodayDischarges($hospitalId)),
        ];
    }

    /**
     * Get recent appointments
     */
    public function getRecentAppointments(int $limit = 10, ?int $hospitalId = null): array
    {
        try {
            $apptModel = new Appointment();
            return $apptModel->recent($limit, $hospitalId);
        } catch (Throwable $e) {
            return [];
        }
    }

    /**
     * Get hospital info by ID
     */
    public function getHospital(int $hospitalId): ?array
    {
        try {
            $hospitalModel = new Hospital();
            return $hospitalModel->find($hospitalId);
        } catch (Throwable $e) {
            return null;
        }
    }

    /**
     * Safe count wrapper - returns 0 on error
     */
    protected function safeCount(callable $fn): int
    {
        try {
            return (int)$fn();
        } catch (Throwable $e) {
            return 0;
        }
    }
}

