<?php
declare(strict_types=1);

class Doctor extends Model
{
    public function all(?int $hospitalId = null): array
    {
        $sql = "SELECT d.id, d.first_name, d.last_name, d.phone, d.email, d.specialty, dept.name AS department, d.created_at
                FROM doctors d
                LEFT JOIN departments dept ON dept.id = d.department_id";
        if ($hospitalId !== null) {
            $sql .= " WHERE d.hospital_id = :hospital_id";
        }
        $sql .= " ORDER BY d.id DESC";
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare($sql);
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query($sql);
        }
        return $stmt->fetchAll();
    }

    public function list(array $filters = []): array
    {
        $q = $filters['q'] ?? '';
        $departmentId = $filters['department_id'] ?? null;
        $hospitalId = $filters['hospital_id'] ?? null;
        $limit = $filters['limit'] ?? 20;
        $offset = $filters['offset'] ?? 0;

        $where = [];
        $params = [];

        $sql = "SELECT d.id, d.first_name, d.last_name, d.phone, d.email, d.specialty, 
                       dept.id AS department_id, dept.name AS department_name, d.created_at
                FROM doctors d
                LEFT JOIN departments dept ON dept.id = d.department_id
                WHERE 1=1";

        // Filter by hospital_id if provided
        if ($hospitalId !== null) {
            $where[] = 'd.hospital_id = :hospital_id';
            $params[':hospital_id'] = $hospitalId;
        }

        if ($q !== '') {
            $where[] = '(d.first_name LIKE :q OR d.last_name LIKE :q OR d.specialty LIKE :q OR d.phone LIKE :q OR d.email LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($departmentId !== null) {
            $where[] = 'd.department_id = :department_id';
            $params[':department_id'] = $departmentId;
        }

        if (!empty($where)) {
            $sql .= ' AND ' . implode(' AND ', $where);
        }

        $sql .= " ORDER BY d.last_name, d.first_name LIMIT :limit OFFSET :offset";

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function count(array $filters = []): int
    {
        $q = $filters['q'] ?? '';
        $departmentId = $filters['department_id'] ?? null;
        $hospitalId = $filters['hospital_id'] ?? null;

        $where = [];
        $params = [];

        $sql = "SELECT COUNT(*) FROM doctors d WHERE 1=1";

        // Filter by hospital_id if provided
        if ($hospitalId !== null) {
            $where[] = 'd.hospital_id = :hospital_id';
            $params[':hospital_id'] = $hospitalId;
        }

        if ($q !== '') {
            $where[] = '(d.first_name LIKE :q OR d.last_name LIKE :q OR d.specialty LIKE :q OR d.phone LIKE :q OR d.email LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($departmentId !== null) {
            $where[] = 'd.department_id = :department_id';
            $params[':department_id'] = $departmentId;
        }

        if (!empty($where)) {
            $sql .= ' AND ' . implode(' AND ', $where);
        }

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function find(int $id, ?int $hospitalId = null): ?array
    {
        $sql = "SELECT d.*, dept.name AS department_name, dept.code AS department_code
                FROM doctors d
                LEFT JOIN departments dept ON dept.id = d.department_id
                WHERE d.id = :id";
        if ($hospitalId !== null) {
            $sql .= " AND d.hospital_id = :hospital_id";
        }
        $stmt = $this->db()->prepare($sql);
        $params = [':id' => $id];
        if ($hospitalId !== null) {
            $params[':hospital_id'] = $hospitalId;
        }
        $stmt->execute($params);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $sql = "INSERT INTO doctors (hospital_id, first_name, last_name, specialty, department_id, phone, email, created_at)
                VALUES (:hospital_id, :first_name, :last_name, :specialty, :department_id, :phone, :email, NOW())";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':hospital_id' => $data['hospital_id'] ?? null,
            ':first_name' => $data['first_name'] ?? null,
            ':last_name'  => $data['last_name'] ?? null,
            ':specialty'  => $data['specialty'] ?? null,
            ':department_id' => $data['department_id'] ?? null,
            ':phone'      => $data['phone'] ?? null,
            ':email'      => $data['email'] ?? null,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function departments(): array
    {
        $stmt = $this->db()->query("SELECT id, name FROM departments WHERE status = 1 ORDER BY name");
        return $stmt->fetchAll();
    }

    public function countAll(?int $hospitalId = null): int
    {
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare('SELECT COUNT(*) FROM doctors WHERE hospital_id = :hospital_id');
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query('SELECT COUNT(*) FROM doctors');
        }
        return (int)$stmt->fetchColumn();
    }

    public function getSchedules(?int $doctorId = null): array
    {
        $sql = "SELECT ds.*, d.first_name, d.last_name, d.specialty
                FROM doctor_schedules ds
                INNER JOIN doctors d ON d.id = ds.doctor_id";
        
        $params = [];
        if ($doctorId !== null) {
            $sql .= " WHERE ds.doctor_id = :doctor_id";
            $params[':doctor_id'] = $doctorId;
        }
        
        $sql .= " ORDER BY ds.day_of_week, ds.start_time";
        
        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function createSchedule(array $data): int
    {
        $sql = "INSERT INTO doctor_schedules (doctor_id, day_of_week, start_time, end_time, is_available, created_at)
                VALUES (:doctor_id, :day_of_week, :start_time, :end_time, :is_available, NOW())";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':doctor_id' => $data['doctor_id'],
            ':day_of_week' => $data['day_of_week'],
            ':start_time' => $data['start_time'],
            ':end_time' => $data['end_time'],
            ':is_available' => $data['is_available'] ?? 1,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function deleteSchedule(int $id): void
    {
        $stmt = $this->db()->prepare("DELETE FROM doctor_schedules WHERE id = :id");
        $stmt->execute([':id' => $id]);
    }

    /* ===================== Attendance ===================== */
    protected function ensureAttendanceTable(): void
    {
        $this->db()->exec("CREATE TABLE IF NOT EXISTS doctor_attendance (
            id INT AUTO_INCREMENT PRIMARY KEY,
            doctor_id INT NOT NULL,
            att_date DATE NOT NULL,
            status VARCHAR(20) DEFAULT 'Full Day',
            check_in TIME NULL,
            check_out TIME NULL,
            overtime_minutes INT DEFAULT 0,
            remarks TEXT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX (doctor_id),
            INDEX (att_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
    }

    public function listAttendance(array $filters = []): array
    {
        $this->ensureAttendanceTable();
        $doctorId = $filters['doctor_id'] ?? null;
        $dateFrom = $filters['date_from'] ?? null;
        $dateTo   = $filters['date_to'] ?? null;
        $limit    = $filters['limit'] ?? 20;
        $offset   = $filters['offset'] ?? 0;

        $sql = "SELECT a.*, d.first_name, d.last_name, d.specialty
                FROM doctor_attendance a
                INNER JOIN doctors d ON d.id = a.doctor_id
                WHERE 1=1";
        $params = [];
        if (!empty($doctorId)) { $sql .= " AND a.doctor_id = :doctor_id"; $params[':doctor_id'] = (int)$doctorId; }
        if (!empty($dateFrom)) { $sql .= " AND a.att_date >= :date_from"; $params[':date_from'] = $dateFrom; }
        if (!empty($dateTo))   { $sql .= " AND a.att_date <= :date_to";   $params[':date_to']   = $dateTo; }

        $sql .= " ORDER BY a.att_date DESC, d.last_name ASC LIMIT :limit OFFSET :offset";
        $stmt = $this->db()->prepare($sql);
        foreach ($params as $k => $v) { $stmt->bindValue($k, $v); }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function countAttendance(array $filters = []): int
    {
        $this->ensureAttendanceTable();
        $doctorId = $filters['doctor_id'] ?? null;
        $dateFrom = $filters['date_from'] ?? null;
        $dateTo   = $filters['date_to'] ?? null;

        $sql = "SELECT COUNT(*) FROM doctor_attendance a WHERE 1=1";
        $params = [];
        if (!empty($doctorId)) { $sql .= " AND a.doctor_id = :doctor_id"; $params[':doctor_id'] = (int)$doctorId; }
        if (!empty($dateFrom)) { $sql .= " AND a.att_date >= :date_from"; $params[':date_from'] = $dateFrom; }
        if (!empty($dateTo))   { $sql .= " AND a.att_date <= :date_to";   $params[':date_to']   = $dateTo; }

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $k => $v) { $stmt->bindValue($k, $v); }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function createAttendance(array $data): int
    {
        $this->ensureAttendanceTable();
        $stmt = $this->db()->prepare("INSERT INTO doctor_attendance (doctor_id, att_date, status, check_in, check_out, overtime_minutes, remarks) VALUES (:doctor_id, :att_date, :status, :check_in, :check_out, :overtime_minutes, :remarks)");
        $stmt->execute([
            ':doctor_id' => (int)$data['doctor_id'],
            ':att_date' => $data['att_date'],
            ':status' => $data['status'] ?? 'Full Day',
            ':check_in' => $data['check_in'] ?: null,
            ':check_out' => $data['check_out'] ?: null,
            ':overtime_minutes' => (int)($data['overtime_minutes'] ?? 0),
            ':remarks' => $data['remarks'] ?? null,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function deleteAttendance(int $id): void
    {
        $this->ensureAttendanceTable();
        $stmt = $this->db()->prepare('DELETE FROM doctor_attendance WHERE id = :id');
        $stmt->execute([':id' => $id]);
    }
}

