<?php
declare(strict_types=1);

class Patient extends Model
{
    public function all(?int $hospitalId = null): array
    {
        $sql = "SELECT id, first_name, last_name, gender, dob, phone, email, created_at FROM patients";
        if ($hospitalId !== null) {
            $sql .= " WHERE hospital_id = :hospital_id";
        }
        $sql .= " ORDER BY id DESC";
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare($sql);
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query($sql);
        }
        return $stmt->fetchAll();
    }

    public function list(array $filters = []): array
    {
        $q = $filters['q'] ?? '';
        $gender = $filters['gender'] ?? null;
        $patientType = $filters['patient_type'] ?? null;
        $status = $filters['status'] ?? null;
        $hospitalId = $filters['hospital_id'] ?? null;
        $limit = $filters['limit'] ?? 20;
        $offset = $filters['offset'] ?? 0;

        $sql = "SELECT 
                    p.id,
                    p.first_name,
                    p.last_name,
                    p.gender,
                    p.dob,
                    p.phone,
                    p.email,
                    p.created_at,
                    -- Admission info (IPD)
                    a.id AS admission_id,
                    a.admitted_at,
                    a.discharged_at,
                    a.diagnosis,
                    -- Bed info
                    b.id AS bed_id,
                    b.ward,
                    b.number AS bed_number,
                    r.name AS room_name,
                    r.code AS room_code,
                    -- Doctor from admission
                    d_adm.id AS doctor_id_adm,
                    CONCAT(d_adm.first_name, ' ', d_adm.last_name) AS doctor_name_adm,
                    -- Appointment info (OPD)
                    apt.id AS appointment_id,
                    apt.scheduled_at AS appointment_date,
                    apt.status AS appointment_status,
                    -- Doctor from appointment
                    d_apt.id AS doctor_id_apt,
                    CONCAT(d_apt.first_name, ' ', d_apt.last_name) AS doctor_name_apt,
                    -- Department from appointment doctor
                    dept_apt.id AS department_id_apt,
                    dept_apt.name AS department_name_apt,
                    -- Department from admission doctor
                    dept_adm.id AS department_id_adm,
                    dept_adm.name AS department_name_adm
                FROM patients p
                LEFT JOIN admissions a ON a.patient_id = p.id AND a.discharged_at IS NULL
                LEFT JOIN beds b ON b.id = a.bed_id
                LEFT JOIN rooms r ON r.id = b.room_id
                LEFT JOIN doctors d_adm ON d_adm.id = a.doctor_id
                LEFT JOIN departments dept_adm ON dept_adm.id = d_adm.department_id
                LEFT JOIN appointments apt ON apt.patient_id = p.id AND apt.status IN ('Booked', 'Completed') 
                    AND apt.scheduled_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                LEFT JOIN doctors d_apt ON d_apt.id = apt.doctor_id
                LEFT JOIN departments dept_apt ON dept_apt.id = d_apt.department_id
                WHERE 1=1";
        $params = [];

        // Filter by hospital_id if provided
        if ($hospitalId !== null) {
            $sql .= " AND p.hospital_id = :hospital_id";
            $params[':hospital_id'] = $hospitalId;
        }

        if ($q !== '') {
            $sql .= " AND (p.first_name LIKE :q OR p.last_name LIKE :q OR p.phone LIKE :q OR p.email LIKE :q)";
            $params[':q'] = '%' . $q . '%';
        }

        if ($gender !== null && $gender !== '') {
            $sql .= " AND p.gender = :gender";
            $params[':gender'] = $gender;
        }

        // Filter by patient type
        if ($patientType === 'IPD') {
            $sql .= " AND a.id IS NOT NULL";
        } elseif ($patientType === 'OPD') {
            $sql .= " AND a.id IS NULL AND apt.id IS NOT NULL";
        } elseif ($patientType === 'Emergency') {
            $sql .= " AND (r.room_type = 'Emergency' OR apt.status = 'Booked')";
        }

        // Filter by status
        if ($status === 'Active') {
            $sql .= " AND a.discharged_at IS NULL";
        } elseif ($status === 'Discharged') {
            $sql .= " AND a.discharged_at IS NOT NULL";
        }

        $sql .= " GROUP BY p.id ORDER BY p.id DESC LIMIT :limit OFFSET :offset";
        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Process results to determine patient type and status
        foreach ($results as &$row) {
            // Determine patient type
            if (!empty($row['admission_id'])) {
                $row['patient_type'] = 'IPD';
                $row['doctor_name'] = $row['doctor_name_adm'] ?? null;
                $row['department_name'] = $row['department_name_adm'] ?? null;
                $row['admission_date'] = $row['admitted_at'];
            } elseif (!empty($row['appointment_id'])) {
                $row['patient_type'] = 'OPD';
                $row['doctor_name'] = $row['doctor_name_apt'] ?? null;
                $row['department_name'] = $row['department_name_apt'] ?? null;
                $row['admission_date'] = null;
            } else {
                $row['patient_type'] = 'OPD';
                $row['doctor_name'] = null;
                $row['department_name'] = null;
                $row['admission_date'] = null;
            }

            // Determine status
            if (!empty($row['admission_id'])) {
                $row['status'] = empty($row['discharged_at']) ? 'Admitted' : 'Discharged';
            } else {
                $row['status'] = 'Active';
            }

            // Format bed info
            if (!empty($row['bed_number'])) {
                $bedInfo = $row['ward'] ?? '';
                if (!empty($row['room_code'])) {
                    $bedInfo .= '-' . $row['room_code'];
                }
                $bedInfo .= '-' . $row['bed_number'];
                $row['bed_info'] = $bedInfo;
            } else {
                $row['bed_info'] = null;
            }
        }
        unset($row);

        return $results;
    }

    public function count(array $filters = []): int
    {
        $q = $filters['q'] ?? '';
        $gender = $filters['gender'] ?? null;
        $patientType = $filters['patient_type'] ?? null;
        $status = $filters['status'] ?? null;
        $hospitalId = $filters['hospital_id'] ?? null;

        $sql = "SELECT COUNT(DISTINCT p.id) 
                FROM patients p
                LEFT JOIN admissions a ON a.patient_id = p.id AND a.discharged_at IS NULL
                LEFT JOIN beds b ON b.id = a.bed_id
                LEFT JOIN rooms r ON r.id = b.room_id
                LEFT JOIN appointments apt ON apt.patient_id = p.id AND apt.status IN ('Booked', 'Completed') 
                    AND apt.scheduled_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                WHERE 1=1";
        $params = [];

        // Filter by hospital_id if provided
        if ($hospitalId !== null) {
            $sql .= " AND p.hospital_id = :hospital_id";
            $params[':hospital_id'] = $hospitalId;
        }

        if ($q !== '') {
            $sql .= " AND (p.first_name LIKE :q OR p.last_name LIKE :q OR p.phone LIKE :q OR p.email LIKE :q)";
            $params[':q'] = '%' . $q . '%';
        }

        if ($gender !== null && $gender !== '') {
            $sql .= " AND p.gender = :gender";
            $params[':gender'] = $gender;
        }

        if ($patientType === 'IPD') {
            $sql .= " AND a.id IS NOT NULL";
        } elseif ($patientType === 'OPD') {
            $sql .= " AND a.id IS NULL AND apt.id IS NOT NULL";
        } elseif ($patientType === 'Emergency') {
            $sql .= " AND (r.room_type = 'Emergency' OR apt.status = 'Booked')";
        }

        if ($status === 'Active') {
            $sql .= " AND a.discharged_at IS NULL";
        } elseif ($status === 'Discharged') {
            $sql .= " AND a.discharged_at IS NOT NULL";
        }

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function find(int $id, ?int $hospitalId = null): ?array
    {
        $sql = "SELECT * FROM patients WHERE id = :id";
        if ($hospitalId !== null) {
            $sql .= " AND hospital_id = :hospital_id";
        }
        $stmt = $this->db()->prepare($sql);
        $params = [':id' => $id];
        if ($hospitalId !== null) {
            $params[':hospital_id'] = $hospitalId;
        }
        $stmt->execute($params);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $sql = "INSERT INTO patients (hospital_id, first_name, last_name, gender, dob, phone, email, address, created_at) VALUES (:hospital_id, :first_name, :last_name, :gender, :dob, :phone, :email, :address, NOW())";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':hospital_id' => $data['hospital_id'] ?? null,
            ':first_name' => $data['first_name'] ?? null,
            ':last_name'  => $data['last_name'] ?? null,
            ':gender'     => $data['gender'] ?? null,
            ':dob'        => !empty($data['dob']) ? $data['dob'] : null,
            ':phone'      => $data['phone'] ?? null,
            ':email'      => $data['email'] ?? null,
            ':address'    => $data['address'] ?? null,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $sql = "UPDATE patients SET
                first_name = :first_name,
                last_name = :last_name,
                gender = :gender,
                dob = :dob,
                phone = :phone,
                email = :email,
                address = :address
                WHERE id = :id";
        $stmt = $this->db()->prepare($sql);
        return $stmt->execute([
            ':first_name' => $data['first_name'] ?? null,
            ':last_name'  => $data['last_name'] ?? null,
            ':gender'     => $data['gender'] ?? null,
            ':dob'        => !empty($data['dob']) ? $data['dob'] : null,
            ':phone'      => $data['phone'] ?? null,
            ':email'      => $data['email'] ?? null,
            ':address'    => $data['address'] ?? null,
            ':id' => $id,
        ]);
    }

    public function countAll(?int $hospitalId = null): int
    {
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare('SELECT COUNT(*) FROM patients WHERE hospital_id = :hospital_id');
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query('SELECT COUNT(*) FROM patients');
        }
        return (int)$stmt->fetchColumn();
    }
}

