<?php
declare(strict_types=1);

class Room extends Model
{
    /**
     * Fetch rooms with optional search/status filter, pagination
     * returns array of rows with extra columns: total_beds, available_beds
     */
    public function list(array $opts = []): array
    {
        $q = trim((string)($opts['q'] ?? ''));
        $status = isset($opts['status']) ? (int)$opts['status'] : null;
        $roomType = isset($opts['room_type']) && $opts['room_type'] !== '' ? (string)$opts['room_type'] : null;
        $hospitalId = isset($opts['hospital_id']) ? (int)$opts['hospital_id'] : null;
        $limit = isset($opts['limit']) ? (int)$opts['limit'] : 20;
        $offset = isset($opts['offset']) ? (int)$opts['offset'] : 0;

        $where = [];
        $params = [];
        if ($hospitalId !== null) {
            $where[] = 'r.hospital_id = :hospital_id';
            $params[':hospital_id'] = $hospitalId;
        }
        if ($q !== '') {
            $where[] = '(r.name LIKE :q OR r.code LIKE :q OR r.description LIKE :q OR r.floor LIKE :q OR r.building LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($status !== null) {
            $where[] = 'r.status = :status';
            $params[':status'] = $status;
        }
        if ($roomType !== null) {
            $where[] = 'r.room_type = :room_type';
            $params[':room_type'] = $roomType;
        }
        $whereSql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

        $sql = "SELECT r.*, 
                   (SELECT COUNT(*) FROM beds b WHERE b.room_id = r.id" . ($hospitalId !== null ? " AND b.hospital_id = r.hospital_id" : "") . ") AS total_beds,
                   (SELECT COUNT(*) FROM beds b WHERE b.room_id = r.id AND b.is_available = 1" . ($hospitalId !== null ? " AND b.hospital_id = r.hospital_id" : "") . ") AS available_beds
                FROM rooms r
                $whereSql
                ORDER BY r.building ASC, r.floor ASC, r.name ASC
                LIMIT :limit OFFSET :offset";

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    public function count(array $opts = []): int
    {
        $q = trim((string)($opts['q'] ?? ''));
        $status = isset($opts['status']) ? (int)$opts['status'] : null;
        $roomType = isset($opts['room_type']) && $opts['room_type'] !== '' ? (string)$opts['room_type'] : null;
        $hospitalId = isset($opts['hospital_id']) ? (int)$opts['hospital_id'] : null;

        $where = [];
        $params = [];
        if ($hospitalId !== null) {
            $where[] = 'hospital_id = :hospital_id';
            $params[':hospital_id'] = $hospitalId;
        }
        if ($q !== '') {
            $where[] = '(name LIKE :q OR code LIKE :q OR description LIKE :q OR floor LIKE :q OR building LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($status !== null) {
            $where[] = 'status = :status';
            $params[':status'] = $status;
        }
        if ($roomType !== null) {
            $where[] = 'room_type = :room_type';
            $params[':room_type'] = $roomType;
        }
        $whereSql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

        $sql = "SELECT COUNT(*) FROM rooms $whereSql";
        $stmt = $this->db()->prepare($sql);
        foreach ($params as $k => $v) { 
            $stmt->bindValue($k, $v); 
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function find(int $id, ?int $hospitalId = null): ?array
    {
        $sql = "SELECT r.*, 
                   (SELECT COUNT(*) FROM beds b WHERE b.room_id = r.id" . ($hospitalId !== null ? " AND b.hospital_id = r.hospital_id" : "") . ") AS total_beds,
                   (SELECT COUNT(*) FROM beds b WHERE b.room_id = r.id AND b.is_available = 1" . ($hospitalId !== null ? " AND b.hospital_id = r.hospital_id" : "") . ") AS available_beds
                FROM rooms r
                WHERE r.id = :id";
        if ($hospitalId !== null) {
            $sql .= " AND r.hospital_id = :hospital_id";
        }
        $sql .= " LIMIT 1";
        $stmt = $this->db()->prepare($sql);
        $params = [':id' => $id];
        if ($hospitalId !== null) {
            $params[':hospital_id'] = $hospitalId;
        }
        $stmt->execute($params);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $sql = "INSERT INTO rooms (hospital_id, name, code, room_type, floor, building, capacity, charge_per_day, description, status, created_at)
                VALUES (:hospital_id, :name, :code, :room_type, :floor, :building, :capacity, :charge_per_day, :description, :status, NOW())";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':hospital_id' => $data['hospital_id'] ?? null,
            ':name' => $data['name'] ?? null,
            ':code' => $data['code'] ?? null,
            ':room_type' => $data['room_type'] ?? 'General',
            ':floor' => $data['floor'] ?? null,
            ':building' => $data['building'] ?? null,
            ':capacity' => isset($data['capacity']) ? (int)$data['capacity'] : 1,
            ':charge_per_day' => isset($data['charge_per_day']) ? (float)$data['charge_per_day'] : 0.00,
            ':description' => $data['description'] ?? null,
            ':status' => isset($data['status']) ? (int)$data['status'] : 1,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $sql = "UPDATE rooms SET 
                name = :name, 
                code = :code, 
                room_type = :room_type, 
                floor = :floor, 
                building = :building, 
                capacity = :capacity, 
                charge_per_day = :charge_per_day, 
                description = :description, 
                status = :status 
                WHERE id = :id";
        
        $stmt = $this->db()->prepare($sql);
        return $stmt->execute([
            ':name' => $data['name'] ?? null,
            ':code' => $data['code'] ?? null,
            ':room_type' => $data['room_type'] ?? 'General',
            ':floor' => $data['floor'] ?? null,
            ':building' => $data['building'] ?? null,
            ':capacity' => isset($data['capacity']) ? (int)$data['capacity'] : 1,
            ':charge_per_day' => isset($data['charge_per_day']) ? (float)$data['charge_per_day'] : 0.00,
            ':description' => $data['description'] ?? null,
            ':status' => isset($data['status']) ? (int)$data['status'] : 1,
            ':id' => $id,
        ]);
    }

    public function getBeds(int $roomId): array
    {
        $sql = "SELECT b.*, 
                   (SELECT COUNT(*) FROM admissions a WHERE a.bed_id = b.id AND a.discharged_at IS NULL) AS is_occupied
                FROM beds b
                WHERE b.room_id = :room_id
                ORDER BY b.number ASC";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([':room_id' => $roomId]);
        return $stmt->fetchAll();
    }

    public function getRoomTypes(): array
    {
        return ['General', 'Private', 'Semi-Private', 'ICU', 'CCU', 'Emergency', 'Operation Theatre', 'Other'];
    }

    public function all(?int $hospitalId = null): array
    {
        if ($hospitalId !== null) {
            $stmt = $this->db()->prepare("SELECT * FROM rooms WHERE status = 1 AND hospital_id = :hospital_id ORDER BY name ASC");
            $stmt->execute([':hospital_id' => $hospitalId]);
        } else {
            $stmt = $this->db()->query("SELECT * FROM rooms WHERE status = 1 ORDER BY name ASC");
        }
        return $stmt->fetchAll();
    }
}






