<?php
declare(strict_types=1);

class Service extends Model
{
    /**
     * Fetch services with optional search/status filter, pagination
     * returns array of rows with extra columns: department_name
     */
    public function list(array $opts = []): array
    {
        $q = trim((string)($opts['q'] ?? ''));
        $status = isset($opts['status']) ? (int)$opts['status'] : null;
        $departmentId = isset($opts['department_id']) ? (int)$opts['department_id'] : null;
        $limit = isset($opts['limit']) ? (int)$opts['limit'] : 20;
        $offset = isset($opts['offset']) ? (int)$opts['offset'] : 0;

        $where = [];
        $params = [];
        if ($q !== '') {
            $where[] = '(s.name LIKE :q OR s.code LIKE :q OR s.description LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($status !== null) {
            $where[] = 's.status = :status';
            $params[':status'] = $status;
        }
        if ($departmentId !== null) {
            $where[] = 's.department_id = :department_id';
            $params[':department_id'] = $departmentId;
        }
        $whereSql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

        $sql = "SELECT s.*, 
                   d.name AS department_name
                FROM services s
                LEFT JOIN departments d ON d.id = s.department_id
                $whereSql
                ORDER BY s.name ASC
                LIMIT :limit OFFSET :offset";

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue($k, $v);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    public function count(array $opts = []): int
    {
        $q = trim((string)($opts['q'] ?? ''));
        $status = isset($opts['status']) ? (int)$opts['status'] : null;
        $departmentId = isset($opts['department_id']) ? (int)$opts['department_id'] : null;

        $where = [];
        $params = [];
        if ($q !== '') {
            $where[] = '(name LIKE :q OR code LIKE :q OR description LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($status !== null) {
            $where[] = 'status = :status';
            $params[':status'] = $status;
        }
        if ($departmentId !== null) {
            $where[] = 'department_id = :department_id';
            $params[':department_id'] = $departmentId;
        }
        $whereSql = $where ? 'WHERE ' . implode(' AND ', $where) : '';

        $sql = "SELECT COUNT(*) FROM services $whereSql";
        $stmt = $this->db()->prepare($sql);
        foreach ($params as $k => $v) { 
            $stmt->bindValue($k, $v); 
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function find(int $id): ?array
    {
        $sql = "SELECT s.*, 
                   d.name AS department_name
                FROM services s
                LEFT JOIN departments d ON d.id = s.department_id
                WHERE s.id = :id
                LIMIT 1";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([':id' => $id]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $sql = "INSERT INTO services (name, code, department_id, description, price, status, created_at)
                VALUES (:name, :code, :department_id, :description, :price, :status, NOW())";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':name' => $data['name'] ?? null,
            ':code' => $data['code'] ?? null,
            ':department_id' => isset($data['department_id']) && (int)$data['department_id'] > 0 ? (int)$data['department_id'] : null,
            ':description' => $data['description'] ?? null,
            ':price' => isset($data['price']) ? (float)$data['price'] : 0.00,
            ':status' => isset($data['status']) ? (int)$data['status'] : 1,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $sql = "UPDATE services SET 
                name = :name, 
                code = :code, 
                department_id = :department_id, 
                description = :description, 
                price = :price, 
                status = :status 
                WHERE id = :id";
        
        $stmt = $this->db()->prepare($sql);
        return $stmt->execute([
            ':name' => $data['name'] ?? null,
            ':code' => $data['code'] ?? null,
            ':department_id' => isset($data['department_id']) && (int)$data['department_id'] > 0 ? (int)$data['department_id'] : null,
            ':description' => $data['description'] ?? null,
            ':price' => isset($data['price']) ? (float)$data['price'] : 0.00,
            ':status' => isset($data['status']) ? (int)$data['status'] : 1,
            ':id' => $id,
        ]);
    }

    public function all(): array
    {
        $sql = "SELECT * FROM services WHERE status = 1 ORDER BY name ASC";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }
}






