<?php
declare(strict_types=1);

class Staff extends Model
{
    public function list(array $filters = []): array
    {
        $q = $filters['q'] ?? '';
        $status = $filters['status'] ?? null;
        $departmentId = $filters['department_id'] ?? null;
        $limit = $filters['limit'] ?? 20;
        $offset = $filters['offset'] ?? 0;

        $sql = "SELECT s.*, d.name AS department_name
                FROM staff s
                LEFT JOIN departments d ON d.id = s.department_id
                WHERE 1=1";
        $params = [];

        if ($q !== '') {
            $sql .= " AND (s.first_name LIKE :q OR s.last_name LIKE :q OR s.employee_id LIKE :q OR s.email LIKE :q OR s.phone LIKE :q)";
            $params[':q'] = '%' . $q . '%';
        }

        if ($status !== null) {
            $sql .= " AND s.status = :status";
            $params[':status'] = $status;
        }

        if ($departmentId !== null) {
            $sql .= " AND s.department_id = :dept";
            $params[':dept'] = $departmentId;
        }

        $sql .= " ORDER BY s.last_name, s.first_name LIMIT :limit OFFSET :offset";
        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function count(array $filters = []): int
    {
        $q = $filters['q'] ?? '';
        $status = $filters['status'] ?? null;
        $departmentId = $filters['department_id'] ?? null;

        $sql = "SELECT COUNT(*) FROM staff s WHERE 1=1";
        $params = [];

        if ($q !== '') {
            $sql .= " AND (s.first_name LIKE :q OR s.last_name LIKE :q OR s.employee_id LIKE :q OR s.email LIKE :q OR s.phone LIKE :q)";
            $params[':q'] = '%' . $q . '%';
        }

        if ($status !== null) {
            $sql .= " AND s.status = :status";
            $params[':status'] = $status;
        }

        if ($departmentId !== null) {
            $sql .= " AND s.department_id = :dept";
            $params[':dept'] = $departmentId;
        }

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function find(int $id): ?array
    {
        $sql = "SELECT s.*, d.name AS department_name
                FROM staff s
                LEFT JOIN departments d ON d.id = s.department_id
                WHERE s.id = :id";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([':id' => $id]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $sql = "INSERT INTO staff (employee_id, first_name, last_name, gender, dob, phone, email, address, department_id, designation, joining_date, salary, shift, status)
                VALUES (:employee_id, :first_name, :last_name, :gender, :dob, :phone, :email, :address, :department_id, :designation, :joining_date, :salary, :shift, :status)";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':employee_id' => $data['employee_id'] ?? null,
            ':first_name' => $data['first_name'] ?? null,
            ':last_name' => $data['last_name'] ?? null,
            ':gender' => $data['gender'] ?? null,
            ':dob' => !empty($data['dob']) ? $data['dob'] : null,
            ':phone' => $data['phone'] ?? null,
            ':email' => $data['email'] ?? null,
            ':address' => $data['address'] ?? null,
            ':department_id' => isset($data['department_id']) && (int)$data['department_id'] > 0 ? (int)$data['department_id'] : null,
            ':designation' => $data['designation'] ?? null,
            ':joining_date' => !empty($data['joining_date']) ? $data['joining_date'] : null,
            ':salary' => isset($data['salary']) ? (float)$data['salary'] : 0.00,
            ':shift' => $data['shift'] ?? 'General',
            ':status' => isset($data['status']) ? (int)$data['status'] : 1,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $sql = "UPDATE staff SET
                employee_id = :employee_id,
                first_name = :first_name,
                last_name = :last_name,
                gender = :gender,
                dob = :dob,
                phone = :phone,
                email = :email,
                address = :address,
                department_id = :department_id,
                designation = :designation,
                joining_date = :joining_date,
                salary = :salary,
                shift = :shift,
                status = :status
                WHERE id = :id";
        $stmt = $this->db()->prepare($sql);
        return $stmt->execute([
            ':employee_id' => $data['employee_id'] ?? null,
            ':first_name' => $data['first_name'] ?? null,
            ':last_name' => $data['last_name'] ?? null,
            ':gender' => $data['gender'] ?? null,
            ':dob' => !empty($data['dob']) ? $data['dob'] : null,
            ':phone' => $data['phone'] ?? null,
            ':email' => $data['email'] ?? null,
            ':address' => $data['address'] ?? null,
            ':department_id' => isset($data['department_id']) && (int)$data['department_id'] > 0 ? (int)$data['department_id'] : null,
            ':designation' => $data['designation'] ?? null,
            ':joining_date' => !empty($data['joining_date']) ? $data['joining_date'] : null,
            ':salary' => isset($data['salary']) ? (float)$data['salary'] : 0.00,
            ':shift' => $data['shift'] ?? 'General',
            ':status' => isset($data['status']) ? (int)$data['status'] : 1,
            ':id' => $id,
        ]);
    }

    public function all(): array
    {
        $sql = "SELECT s.id, s.employee_id, s.first_name, s.last_name, s.designation, d.name AS department_name, s.status
                FROM staff s
                LEFT JOIN departments d ON d.id = s.department_id
                WHERE s.status = 1
                ORDER BY s.last_name, s.first_name";
        $stmt = $this->db()->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getDepartments(): array
    {
        $stmt = $this->db()->query("SELECT id, name FROM departments WHERE status = 1 ORDER BY name");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function generateEmployeeId(): string
    {
        $stmt = $this->db()->query("SELECT COUNT(*) + 1 AS next_num FROM staff");
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $num = (int)($row['next_num'] ?? 1);
        return 'EMP' . str_pad((string)$num, 5, '0', STR_PAD_LEFT);
    }
}






