<?php
/**
 * Script to create admin users for hospitals that don't have one
 * Run via browser: http://localhost/hms_project/create_missing_hospital_admins.php
 */

// Load database configuration
$dbConfigFile = __DIR__ . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'database.php';
if (!file_exists($dbConfigFile)) {
    die("❌ Database config file not found at: " . htmlspecialchars($dbConfigFile));
}

$dbConfig = require $dbConfigFile;

// Verify config loaded correctly
if (!is_array($dbConfig)) {
    die("❌ Database config file did not return an array");
}

// Extract connection details
$host = '127.0.0.1';
$dbname = 'hms';
$user = isset($dbConfig['user']) ? $dbConfig['user'] : 'root';
$pass = isset($dbConfig['pass']) ? $dbConfig['pass'] : '';

echo "<!DOCTYPE html><html><head><title>Create Missing Hospital Admins</title>";
echo "<style>
    body{font-family:Arial;padding:20px;background:#f5f5f5;}
    .container{max-width:900px;margin:0 auto;background:#fff;padding:20px;border-radius:8px;box-shadow:0 2px 10px rgba(0,0,0,0.1);}
    h1{color:#333;border-bottom:2px solid #007bff;padding-bottom:10px;}
    table{border-collapse:collapse;width:100%;margin:20px 0;}
    th,td{padding:12px;text-align:left;border:1px solid #ddd;}
    th{background:#007bff;color:#fff;}
    tr:nth-child(even){background:#f9f9f9;}
    .success{color:#28a745;font-weight:bold;}
    .error{color:#dc3545;font-weight:bold;}
    .info{color:#17a2b8;background:#d1ecf1;padding:15px;border-radius:5px;margin:20px 0;}
    .credentials{background:#fff3cd;padding:10px;border-left:4px solid #ffc107;margin:10px 0;}
</style></head><body>";
echo "<div class='container'>";
echo "<h1>🏥 Create Missing Hospital Admin Users</h1>";

try {
    $pdo = new PDO(
        "mysql:host={$host};dbname={$dbname};charset=utf8mb4",
        $user,
        $pass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );

    // Get all hospitals
    $stmt = $pdo->query("SELECT id, name, code FROM hospitals ORDER BY id");
    $hospitals = $stmt->fetchAll();
    
    if (empty($hospitals)) {
        echo "<p class='error'>No hospitals found. Please add hospitals first.</p>";
        echo "</div></body></html>";
        exit;
    }

    echo "<div class='info'>";
    echo "This script will create admin users for hospitals that don't have one.<br/>";
    echo "Default password for all new admins: <strong>admin123</strong>";
    echo "</div>";

    $created = [];
    $skipped = [];
    $errors = [];

    echo "<h2>Processing Hospitals:</h2>";
    echo "<table><tr><th>Hospital ID</th><th>Hospital Name</th><th>Code</th><th>Status</th><th>Email Created</th></tr>";

    foreach ($hospitals as $hospital) {
        $hospitalId = $hospital['id'];
        $hospitalName = $hospital['name'];
        $hospitalCode = trim($hospital['code'] ?? '');
        
        // Generate email
        if (empty($hospitalCode)) {
            $emailCode = 'hospital' . $hospitalId;
        } else {
            $emailCode = strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $hospitalCode));
            if (empty($emailCode)) {
                $emailCode = 'hospital' . $hospitalId;
            }
        }
        
        $defaultEmail = $emailCode . '@hospital.local';
        $defaultPassword = 'admin123';
        $defaultName = $hospitalName . ' Administrator';
        
        // Check if admin already exists for this hospital
        $stmt = $pdo->prepare("SELECT id, email FROM admins WHERE hospital_id = ? AND role = 'admin' LIMIT 1");
        $stmt->execute([$hospitalId]);
        $existingAdmin = $stmt->fetch();
        
        if ($existingAdmin) {
            echo "<tr><td>{$hospitalId}</td><td>{$hospitalName}</td><td>{$hospitalCode}</td>";
            echo "<td class='info'>✓ Already has admin</td>";
            echo "<td>{$existingAdmin['email']}</td></tr>";
            $skipped[] = ['hospital' => $hospitalName, 'email' => $existingAdmin['email']];
        } else {
            // Check if email already exists (might be assigned to different hospital)
            $stmt = $pdo->prepare("SELECT id FROM admins WHERE email = ? LIMIT 1");
            $stmt->execute([$defaultEmail]);
            $emailExists = $stmt->fetch();
            
            if ($emailExists) {
                // Email exists but for different hospital, use hospital ID in email
                $defaultEmail = 'hospital' . $hospitalId . '@hospital.local';
            }
            
            try {
                // Create admin user
                $passwordHash = password_hash($defaultPassword, PASSWORD_BCRYPT);
                $stmt = $pdo->prepare("
                    INSERT INTO admins (name, email, role, hospital_id, password_hash, created_at) 
                    VALUES (:name, :email, 'admin', :hospital_id, :password_hash, NOW())
                ");
                $stmt->execute([
                    ':name' => $defaultName,
                    ':email' => $defaultEmail,
                    ':hospital_id' => $hospitalId,
                    ':password_hash' => $passwordHash,
                ]);
                
                $adminId = $pdo->lastInsertId();
                
                echo "<tr><td>{$hospitalId}</td><td>{$hospitalName}</td><td>{$hospitalCode}</td>";
                echo "<td class='success'>✓ Created</td>";
                echo "<td>{$defaultEmail}</td></tr>";
                
                $created[] = [
                    'hospital' => $hospitalName,
                    'email' => $defaultEmail,
                    'password' => $defaultPassword,
                    'admin_id' => $adminId
                ];
            } catch (PDOException $e) {
                echo "<tr><td>{$hospitalId}</td><td>{$hospitalName}</td><td>{$hospitalCode}</td>";
                echo "<td class='error'>✗ Error</td>";
                echo "<td>" . htmlspecialchars($e->getMessage()) . "</td></tr>";
                $errors[] = ['hospital' => $hospitalName, 'error' => $e->getMessage()];
            }
        }
    }
    
    echo "</table>";

    // Show created credentials
    if (!empty($created)) {
        echo "<h2>✅ Created Admin Users - Login Credentials:</h2>";
        echo "<div class='credentials'>";
        echo "<table><tr><th>Hospital</th><th>Email</th><th>Password</th></tr>";
        foreach ($created as $admin) {
            echo "<tr>";
            echo "<td>{$admin['hospital']}</td>";
            echo "<td><strong>{$admin['email']}</strong></td>";
            echo "<td><strong>{$admin['password']}</strong></td>";
            echo "</tr>";
        }
        echo "</table>";
        echo "</div>";
        echo "<p class='info'><strong>⚠ Important:</strong> Please note these credentials. Users should change their password after first login.</p>";
    }

    if (!empty($skipped)) {
        echo "<h2>ℹ Skipped (Already have admin):</h2>";
        echo "<ul>";
        foreach ($skipped as $skip) {
            echo "<li>{$skip['hospital']} - {$skip['email']}</li>";
        }
        echo "</ul>";
    }

    if (!empty($errors)) {
        echo "<h2>❌ Errors:</h2>";
        echo "<ul>";
        foreach ($errors as $error) {
            echo "<li>{$error['hospital']}: {$error['error']}</li>";
        }
        echo "</ul>";
    }

    echo "<p><a href='admin/'>Go to Admin Panel</a> | <a href='superadmin/'>Go to Super Admin Panel</a></p>";

} catch (PDOException $e) {
    echo "<p class='error'>Database Error: " . htmlspecialchars($e->getMessage()) . "</p>";
} catch (Exception $e) {
    echo "<p class='error'>Error: " . htmlspecialchars($e->getMessage()) . "</p>";
}

echo "</div>";
echo "</body></html>";

