-- Multi-Hospital Support Database Schema
-- Run this after initial database setup

USE `hms`;

-- Hospitals table
CREATE TABLE IF NOT EXISTS `hospitals` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `name` VARCHAR(200) NOT NULL,
  `code` VARCHAR(50) NULL,
  `address` TEXT NULL,
  `city` VARCHAR(100) NULL,
  `state` VARCHAR(100) NULL,
  `country` VARCHAR(100) NULL DEFAULT 'India',
  `pincode` VARCHAR(20) NULL,
  `phone` VARCHAR(20) NULL,
  `email` VARCHAR(150) NULL,
  `website` VARCHAR(255) NULL,
  `registration_number` VARCHAR(100) NULL,
  `license_number` VARCHAR(100) NULL,
  `logo` VARCHAR(255) NULL,
  `status` ENUM('Active','Inactive','Suspended') NOT NULL DEFAULT 'Active',
  `subscription_plan` VARCHAR(50) NULL DEFAULT 'Basic',
  `subscription_expires_at` DATETIME NULL,
  `max_users` INT UNSIGNED NULL DEFAULT 10,
  `max_patients` INT UNSIGNED NULL,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uq_hospitals_code` (`code`),
  KEY `idx_hospitals_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Add hospital_id to admins table (NULL for super admin)
ALTER TABLE `admins` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `role`,
ADD KEY IF NOT EXISTS `idx_admins_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_admins_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE SET NULL;

-- Add hospital_id to all hospital-specific tables
-- Patients
ALTER TABLE `patients` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_patients_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_patients_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Departments
ALTER TABLE `departments` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_departments_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_departments_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Doctors
ALTER TABLE `doctors` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_doctors_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_doctors_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Appointments
ALTER TABLE `appointments` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_appointments_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_appointments_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Rooms
ALTER TABLE `rooms` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_rooms_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_rooms_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Beds
ALTER TABLE `beds` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_beds_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_beds_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Admissions
ALTER TABLE `admissions` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_admissions_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_admissions_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Invoices
ALTER TABLE `invoices` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_invoices_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_invoices_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Medicines
ALTER TABLE `medicines` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_medicines_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_medicines_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Pharmacy Sales
ALTER TABLE `pharmacy_sales` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_pharmacy_sales_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_pharmacy_sales_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Lab Tests
ALTER TABLE `lab_tests` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_lab_tests_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_lab_tests_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Lab Orders
ALTER TABLE `lab_orders` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_lab_orders_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_lab_orders_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Services
ALTER TABLE `services` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_services_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_services_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Charges
ALTER TABLE `charges` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_charges_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_charges_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- HR Staff
ALTER TABLE `hr_staff` 
ADD COLUMN IF NOT EXISTS `hospital_id` INT UNSIGNED NULL AFTER `id`,
ADD KEY IF NOT EXISTS `idx_hr_staff_hospital` (`hospital_id`),
ADD CONSTRAINT IF NOT EXISTS `fk_hr_staff_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE;

-- Doctor Schedules
CREATE TABLE IF NOT EXISTS `doctor_schedules` (
  `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `hospital_id` INT UNSIGNED NULL,
  `doctor_id` INT UNSIGNED NOT NULL,
  `day_of_week` VARCHAR(20) NOT NULL,
  `start_time` TIME NOT NULL,
  `end_time` TIME NOT NULL,
  `is_available` TINYINT(1) NOT NULL DEFAULT 1,
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_schedules_hospital` (`hospital_id`),
  KEY `idx_schedules_doctor` (`doctor_id`),
  CONSTRAINT `fk_schedules_hospital` FOREIGN KEY (`hospital_id`) REFERENCES `hospitals` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_schedules_doctor` FOREIGN KEY (`doctor_id`) REFERENCES `doctors` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Create default super admin (if not exists)
INSERT IGNORE INTO `admins` (`name`, `email`, `role`, `hospital_id`, `password_hash`, `created_at`) 
VALUES ('Super Administrator', 'superadmin@hms.local', 'super_admin', NULL, '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', NOW());
-- Default password: password (change after first login)

