<?php
/**
 * Complete Debug Script for Super Admin Login
 * This will test everything step by step
 */

error_reporting(E_ALL);
ini_set('display_errors', '1');

echo "<!DOCTYPE html>
<html lang='en'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Super Admin Login Debug</title>
    <link href='https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css' rel='stylesheet'>
    <style>
        body { background: #f5f5f5; padding: 20px; }
        .card { margin-bottom: 20px; }
        .success { color: #28a745; }
        .error { color: #dc3545; }
        .warning { color: #ffc107; }
        pre { background: #f8f9fa; padding: 10px; border-radius: 5px; }
    </style>
</head>
<body>
<div class='container'>
    <div class='card'>
        <div class='card-header bg-primary text-white'>
            <h3>Super Admin Login - Complete Debug</h3>
        </div>
        <div class='card-body'>";

$issues = [];
$success = [];

// Step 1: Check database connection
echo "<h4>Step 1: Database Connection</h4>";
try {
    $dbHost = '127.0.0.1';
    $dbPort = '3306';
    $dbUser = 'root';
    $dbPass = '';
    $dbName = 'hms';
    
    $pdo = new PDO(
        "mysql:host={$dbHost};port={$dbPort};dbname={$dbName};charset=utf8mb4",
        $dbUser,
        $dbPass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    echo "<p class='success'>✓ Database connected successfully</p>";
    $success[] = "Database connection";
} catch (PDOException $e) {
    echo "<p class='error'>✗ Database connection failed: " . htmlspecialchars($e->getMessage()) . "</p>";
    $issues[] = "Database connection failed";
    echo "</div></div></div></body></html>";
    exit;
}

// Step 2: Check admins table
echo "<h4>Step 2: Check Admins Table</h4>";
try {
    $stmt = $pdo->query("SHOW TABLES LIKE 'admins'");
    if ($stmt->rowCount() === 0) {
        echo "<p class='error'>✗ 'admins' table does not exist!</p>";
        $issues[] = "Admins table missing";
    } else {
        echo "<p class='success'>✓ 'admins' table exists</p>";
        $success[] = "Admins table exists";
        
        // Check table structure
        $stmt = $pdo->query("DESCRIBE admins");
        $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
        $requiredColumns = ['id', 'name', 'email', 'role', 'password_hash', 'hospital_id'];
        $missingColumns = array_diff($requiredColumns, $columns);
        
        if (!empty($missingColumns)) {
            echo "<p class='error'>✗ Missing columns: " . implode(', ', $missingColumns) . "</p>";
            $issues[] = "Missing table columns: " . implode(', ', $missingColumns);
        } else {
            echo "<p class='success'>✓ All required columns exist</p>";
        }
    }
} catch (PDOException $e) {
    echo "<p class='error'>✗ Error checking table: " . htmlspecialchars($e->getMessage()) . "</p>";
    $issues[] = "Table check error";
}

// Step 3: Check super admin user
echo "<h4>Step 3: Check Super Admin User</h4>";
$email = 'superadmin@hms.local';
$password = 'password';

try {
    $stmt = $pdo->prepare("SELECT id, name, email, role, hospital_id, password_hash FROM admins WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if (!$user) {
        echo "<p class='error'>✗ Super admin user NOT found!</p>";
        echo "<p class='warning'>Creating super admin user...</p>";
        
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("
            INSERT INTO admins (name, email, role, hospital_id, password_hash, created_at) 
            VALUES (:name, :email, :role, NULL, :password_hash, NOW())
        ");
        $stmt->execute([
            ':name' => 'Super Administrator',
            ':email' => $email,
            ':role' => 'super_admin',
            ':password_hash' => $passwordHash,
        ]);
        
        $userId = $pdo->lastInsertId();
        echo "<p class='success'>✓ Super admin user created! ID: {$userId}</p>";
        $success[] = "Super admin user created";
        
        // Fetch again
        $stmt = $pdo->prepare("SELECT id, name, email, role, hospital_id, password_hash FROM admins WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
    } else {
        echo "<p class='success'>✓ Super admin user found</p>";
        $success[] = "Super admin user exists";
    }
    
    // Display user details
    echo "<pre>";
    echo "User ID: " . ($user['id'] ?? 'N/A') . "\n";
    echo "Name: " . ($user['name'] ?? 'N/A') . "\n";
    echo "Email: " . ($user['email'] ?? 'N/A') . "\n";
    echo "Role: " . ($user['role'] ?? 'N/A') . "\n";
    echo "Hospital ID: " . ($user['hospital_id'] ?? 'NULL') . "\n";
    echo "</pre>";
    
    // Check role
    if (($user['role'] ?? '') !== 'super_admin') {
        echo "<p class='error'>✗ Role is '{$user['role']}' but should be 'super_admin'</p>";
        echo "<p class='warning'>Updating role...</p>";
        $stmt = $pdo->prepare("UPDATE admins SET role = 'super_admin' WHERE id = ?");
        $stmt->execute([$user['id']]);
        echo "<p class='success'>✓ Role updated to 'super_admin'</p>";
        $user['role'] = 'super_admin';
    } else {
        echo "<p class='success'>✓ Role is correct: super_admin</p>";
    }
    
} catch (PDOException $e) {
    echo "<p class='error'>✗ Error: " . htmlspecialchars($e->getMessage()) . "</p>";
    $issues[] = "User check error";
}

// Step 4: Test password verification
echo "<h4>Step 4: Password Verification</h4>";
if (isset($user) && !empty($user['password_hash'])) {
    if (password_verify($password, $user['password_hash'])) {
        echo "<p class='success'>✓ Password verification: SUCCESS</p>";
        $success[] = "Password verification";
    } else {
        echo "<p class='error'>✗ Password verification: FAILED</p>";
        echo "<p class='warning'>Updating password...</p>";
        
        $newHash = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE admins SET password_hash = ? WHERE id = ?");
        $stmt->execute([$newHash, $user['id']]);
        
        // Verify again
        if (password_verify($password, $newHash)) {
            echo "<p class='success'>✓ Password updated and verified successfully</p>";
            $success[] = "Password updated";
        } else {
            echo "<p class='error'>✗ Password update failed</p>";
            $issues[] = "Password update failed";
        }
    }
} else {
    echo "<p class='error'>✗ Cannot verify password - user or hash missing</p>";
    $issues[] = "Password verification failed";
}

// Step 5: Test Admin Model
echo "<h4>Step 5: Test Admin Model</h4>";
try {
    // Check if Admin model file exists
    $adminModelPath = dirname(__DIR__) . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'Admin.php';
    if (!file_exists($adminModelPath)) {
        echo "<p class='error'>✗ Admin.php model file not found at: {$adminModelPath}</p>";
        $issues[] = "Admin model file missing";
    } else {
        echo "<p class='success'>✓ Admin model file exists</p>";
        
        // Try to load and test
        require_once dirname(__DIR__) . DIRECTORY_SEPARATOR . 'system' . DIRECTORY_SEPARATOR . 'Model.php';
        require_once $adminModelPath;
        
        if (class_exists('Admin')) {
            echo "<p class='success'>✓ Admin class loaded</p>";
            
            // Test database connection through model
            $adminModel = new Admin();
            $testUser = $adminModel->findByEmail($email);
            
            if ($testUser) {
                echo "<p class='success'>✓ Admin model can find user</p>";
                echo "<pre>Model returned: " . print_r($testUser, true) . "</pre>";
                
                // Check if role is in result
                if (isset($testUser['role']) && $testUser['role'] === 'super_admin') {
                    echo "<p class='success'>✓ Model returns correct role</p>";
                } else {
                    echo "<p class='error'>✗ Model returns wrong role: " . ($testUser['role'] ?? 'missing') . "</p>";
                    $issues[] = "Model returns wrong role";
                }
            } else {
                echo "<p class='error'>✗ Admin model cannot find user</p>";
                $issues[] = "Model cannot find user";
            }
        } else {
            echo "<p class='error'>✗ Admin class not found</p>";
            $issues[] = "Admin class not found";
        }
    }
} catch (Exception $e) {
    echo "<p class='error'>✗ Error testing model: " . htmlspecialchars($e->getMessage()) . "</p>";
    $issues[] = "Model test error";
}

// Step 6: Test Auth class
echo "<h4>Step 6: Test Auth Class</h4>";
try {
    $authPath = dirname(__DIR__) . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'Auth.php';
    if (!file_exists($authPath)) {
        echo "<p class='error'>✗ Auth.php file not found</p>";
        $issues[] = "Auth file missing";
    } else {
        echo "<p class='success'>✓ Auth file exists</p>";
        require_once $authPath;
        
        if (class_exists('Auth')) {
            echo "<p class='success'>✓ Auth class loaded</p>";
            
            // Test login method
            if (method_exists('Auth', 'login')) {
                echo "<p class='success'>✓ Auth::login() method exists</p>";
            } else {
                echo "<p class='error'>✗ Auth::login() method missing</p>";
                $issues[] = "Auth login method missing";
            }
        } else {
            echo "<p class='error'>✗ Auth class not found</p>";
            $issues[] = "Auth class not found";
        }
    }
} catch (Exception $e) {
    echo "<p class='error'>✗ Error testing Auth: " . htmlspecialchars($e->getMessage()) . "</p>";
    $issues[] = "Auth test error";
}

// Step 7: Test session
echo "<h4>Step 7: Session Test</h4>";
if (session_status() === PHP_SESSION_ACTIVE) {
    echo "<p class='success'>✓ Session is active</p>";
} else {
    session_start();
    echo "<p class='warning'>⚠ Session started manually</p>";
}

// Summary
echo "<hr><h4>Summary</h4>";
echo "<div class='alert alert-success'><strong>Success:</strong> " . count($success) . " checks passed</div>";
if (!empty($issues)) {
    echo "<div class='alert alert-danger'><strong>Issues Found:</strong><ul>";
    foreach ($issues as $issue) {
        echo "<li>" . htmlspecialchars($issue) . "</li>";
    }
    echo "</ul></div>";
} else {
    echo "<div class='alert alert-success'><strong>All checks passed! Login should work now.</strong></div>";
}

// Final credentials
echo "<div class='alert alert-info'>
    <h5>Login Credentials:</h5>
    <p><strong>Email:</strong> {$email}</p>
    <p><strong>Password:</strong> {$password}</p>
    <p><strong>Login URL:</strong> <a href='http://localhost/hms_project/superadmin/login' target='_blank'>http://localhost/hms_project/superadmin/login</a></p>
</div>";

echo "</div></div></div>
<script src='https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js'></script>
</body></html>";
?>

