<?php
/**
 * Debug Super Admin Login Error
 * Run this script to identify the login issue
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h2>Super Admin Login Debug</h2>";
echo "<pre>";

// Step 1: Check database connection
echo "Step 1: Checking database connection...\n";
try {
    // Load database config
    $dbConfigFile = __DIR__ . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'database.php';
    if (!file_exists($dbConfigFile)) {
        throw new Exception("Database config file not found: {$dbConfigFile}");
    }
    
    $dbConfig = require $dbConfigFile;
    
    // Use dsn if available, otherwise construct it
    if (isset($dbConfig['dsn'])) {
        $pdo = new PDO(
            $dbConfig['dsn'],
            $dbConfig['user'] ?? 'root',
            $dbConfig['pass'] ?? '',
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
    } else {
        // Fallback: construct DSN from individual components
        $host = $dbConfig['host'] ?? '127.0.0.1';
        $port = $dbConfig['port'] ?? 3306;
        $dbname = $dbConfig['dbname'] ?? 'hms';
        $user = $dbConfig['user'] ?? 'root';
        $pass = $dbConfig['pass'] ?? '';
        
        $pdo = new PDO(
            "mysql:host={$host};port={$port};dbname={$dbname};charset=utf8mb4",
            $user,
            $pass,
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
    }
    
    echo "✓ Database connection successful\n";
    echo "   Database: " . ($dbConfig['dbname'] ?? 'hms') . "\n";
    echo "   Host: " . ($dbConfig['host'] ?? '127.0.0.1') . "\n";
} catch (Exception $e) {
    echo "✗ Database connection failed: " . $e->getMessage() . "\n";
    echo "   Config file: {$dbConfigFile}\n";
    echo "\n";
    echo "Please check:\n";
    echo "1. MySQL server is running\n";
    echo "2. Database credentials in: admin/app/config/database.php\n";
    echo "3. Database 'hms' exists\n";
    exit;
}

// Step 2: Check database schema first
echo "\nStep 2: Checking database schema...\n";
try {
    $stmt = $pdo->query("SHOW COLUMNS FROM admins LIKE 'hospital_id'");
    $hasHospitalId = $stmt->fetch() !== false;
    
    if (!$hasHospitalId) {
        echo "⚠ hospital_id column not found in admins table\n";
        echo "Adding hospital_id column...\n";
        
        try {
            // Check if hospitals table exists
            $stmt = $pdo->query("SHOW TABLES LIKE 'hospitals'");
            $hospitalsTableExists = $stmt->fetch() !== false;
            
            if ($hospitalsTableExists) {
                $pdo->exec("ALTER TABLE admins ADD COLUMN hospital_id INT UNSIGNED NULL AFTER role");
                $pdo->exec("ALTER TABLE admins ADD KEY idx_admins_hospital (hospital_id)");
                $pdo->exec("ALTER TABLE admins ADD CONSTRAINT fk_admins_hospital FOREIGN KEY (hospital_id) REFERENCES hospitals (id) ON DELETE SET NULL");
                echo "✓ hospital_id column added successfully\n";
            } else {
                // Just add the column without foreign key if hospitals table doesn't exist
                $pdo->exec("ALTER TABLE admins ADD COLUMN hospital_id INT UNSIGNED NULL AFTER role");
                echo "✓ hospital_id column added (without foreign key - hospitals table doesn't exist yet)\n";
            }
        } catch (Exception $e) {
            echo "⚠ Could not add hospital_id column: " . $e->getMessage() . "\n";
            echo "   Continuing with existing schema...\n";
        }
    } else {
        echo "✓ hospital_id column exists\n";
    }
} catch (Exception $e) {
    echo "⚠ Could not check schema: " . $e->getMessage() . "\n";
}

// Step 3: Check if super admin user exists
echo "\nStep 3: Checking super admin user...\n";
$email = 'superadmin@hms.local';

// Check which columns exist
$stmt = $pdo->query("SHOW COLUMNS FROM admins");
$columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
$hasHospitalId = in_array('hospital_id', $columns);

$selectFields = ['id', 'name', 'email', 'role', 'password_hash'];
if ($hasHospitalId) {
    $selectFields[] = 'hospital_id';
}

$sql = "SELECT " . implode(', ', $selectFields) . " FROM admins WHERE email = ?";
$stmt = $pdo->prepare($sql);
$stmt->execute([$email]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$user) {
    echo "✗ Super admin user NOT found!\n";
    echo "Creating super admin user...\n";
    
    $password = 'password';
    $passwordHash = password_hash($password, PASSWORD_DEFAULT);
    
    try {
        // Build INSERT query based on available columns
        $insertFields = ['name', 'email', 'role', 'password_hash', 'created_at'];
        $insertValues = ['Super Administrator', $email, 'super_admin', $passwordHash, date('Y-m-d H:i:s')];
        
        if ($hasHospitalId) {
            $insertFields[] = 'hospital_id';
            $insertValues[] = null;
        }
        
        $placeholders = implode(', ', array_fill(0, count($insertFields), '?'));
        $sql = "INSERT INTO admins (" . implode(', ', $insertFields) . ") VALUES ({$placeholders})";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($insertValues);
        echo "✓ Super admin user created successfully!\n";
        echo "   Email: {$email}\n";
        echo "   Password: {$password}\n";
        
        // Fetch again
        $sql = "SELECT " . implode(', ', $selectFields) . " FROM admins WHERE email = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$email]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        echo "✗ Failed to create super admin: " . $e->getMessage() . "\n";
        exit;
    }
} else {
    echo "✓ Super admin user found\n";
    echo "   ID: {$user['id']}\n";
    echo "   Name: {$user['name']}\n";
    echo "   Email: {$user['email']}\n";
    echo "   Role: {$user['role']}\n";
    if (isset($user['hospital_id'])) {
        echo "   Hospital ID: " . ($user['hospital_id'] ?? 'NULL') . "\n";
    }
}

// Step 4: Verify password
echo "\nStep 4: Verifying password...\n";
$testPassword = 'password';
if (password_verify($testPassword, $user['password_hash'])) {
    echo "✓ Password verification successful\n";
} else {
    echo "✗ Password verification FAILED!\n";
    echo "Updating password hash...\n";
    
    $newHash = password_hash($testPassword, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare("UPDATE admins SET password_hash = ? WHERE id = ?");
    $stmt->execute([$newHash, $user['id']]);
    echo "✓ Password hash updated\n";
}

// Step 5: Check role
echo "\nStep 5: Checking role...\n";
if (($user['role'] ?? '') === 'super_admin') {
    echo "✓ Role is correct: super_admin\n";
} else {
    echo "✗ Role is incorrect: " . ($user['role'] ?? 'none') . "\n";
    echo "Updating role to super_admin...\n";
    
    $stmt = $pdo->prepare("UPDATE admins SET role = 'super_admin' WHERE id = ?");
    $stmt->execute([$user['id']]);
    echo "✓ Role updated to super_admin\n";
}

// Step 6: Test Admin model
echo "\nStep 6: Testing Admin model...\n";
try {
    require_once __DIR__ . DIRECTORY_SEPARATOR . 'system' . DIRECTORY_SEPARATOR . 'Model.php';
    require_once __DIR__ . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'Admin.php';
    
    $adminModel = new Admin();
    $foundUser = $adminModel->findByEmail($email);
    
    if ($foundUser) {
        echo "✓ Admin model findByEmail() works\n";
        echo "   Found user: {$foundUser['name']}\n";
        echo "   Role: " . ($foundUser['role'] ?? 'none') . "\n";
    } else {
        echo "✗ Admin model findByEmail() returned null\n";
    }
} catch (Exception $e) {
    echo "✗ Admin model error: " . $e->getMessage() . "\n";
    echo "   Stack trace:\n";
    echo $e->getTraceAsString() . "\n";
}

// Step 7: Test Auth class
echo "\nStep 7: Testing Auth class...\n";
try {
    require_once __DIR__ . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'Auth.php';
    
    if (class_exists('Auth')) {
        echo "✓ Auth class loaded\n";
    } else {
        echo "✗ Auth class not found\n";
    }
} catch (Exception $e) {
    echo "✗ Auth class error: " . $e->getMessage() . "\n";
}

// Step 8: Test complete login flow
echo "\nStep 8: Testing complete login flow...\n";
try {
    session_start();
    
    $adminModel = new Admin();
    $testUser = $adminModel->findByEmail($email);
    $testPassword = 'password';
    
    if ($testUser && password_verify($testPassword, $testUser['password_hash'])) {
        if (($testUser['role'] ?? '') === 'super_admin') {
            Auth::login($testUser);
            $loggedInUser = Auth::user();
            
            if ($loggedInUser) {
                echo "✓ Complete login flow successful!\n";
                echo "   Logged in user: {$loggedInUser['name']}\n";
                echo "   Role: {$loggedInUser['role']}\n";
            } else {
                echo "✗ Auth::login() worked but Auth::user() returned null\n";
            }
        } else {
            echo "✗ User role is not super_admin\n";
        }
    } else {
        echo "✗ Password verification failed in test\n";
    }
} catch (Exception $e) {
    echo "✗ Login flow error: " . $e->getMessage() . "\n";
    echo "   Stack trace:\n";
    echo $e->getTraceAsString() . "\n";
}

echo "\n";
echo "========================================\n";
echo "Debug Complete!\n";
echo "========================================\n";
echo "\n";
echo "If all checks passed, try logging in again:\n";
echo "URL: http://localhost/hms_project/superadmin/login\n";
echo "Email: superadmin@hms.local\n";
echo "Password: password\n";
echo "</pre>";
?>

