<?php
/**
 * Fix Super Admin Login Issue
 * This script will check and fix the super admin user
 */

error_reporting(E_ALL);
ini_set('display_errors', '1');

// Database configuration
$dbHost = '127.0.0.1';
$dbPort = '3306';
$dbUser = 'root';
$dbPass = '';
$dbName = 'hms';

echo "<h2>Fix Super Admin Login</h2>";
echo "<pre>";

try {
    // Connect to database
    $pdo = new PDO(
        "mysql:host={$dbHost};port={$dbPort};dbname={$dbName};charset=utf8mb4",
        $dbUser,
        $dbPass,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );
    echo "✓ Database connected\n\n";

    $email = 'superadmin@hms.local';
    $password = 'password';
    
    // Check if user exists
    echo "Step 1: Checking if user exists...\n";
    $stmt = $pdo->prepare("SELECT id, name, email, role, hospital_id, password_hash FROM admins WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if (!$user) {
        echo "✗ User NOT found! Creating new user...\n\n";
        
        // Create new user with fresh password hash
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        
        $stmt = $pdo->prepare("
            INSERT INTO admins (name, email, role, hospital_id, password_hash, created_at) 
            VALUES (:name, :email, :role, NULL, :password_hash, NOW())
        ");
        $stmt->execute([
            ':name' => 'Super Administrator',
            ':email' => $email,
            ':role' => 'super_admin',
            ':password_hash' => $passwordHash,
        ]);
        
        $userId = $pdo->lastInsertId();
        echo "✓ Super admin created successfully!\n";
        echo "   User ID: {$userId}\n";
        echo "   Email: {$email}\n";
        echo "   Role: super_admin\n";
        echo "   Password: {$password}\n\n";
        
        // Verify password
        $stmt = $pdo->prepare("SELECT password_hash FROM admins WHERE id = ?");
        $stmt->execute([$userId]);
        $newUser = $stmt->fetch();
        
        if (password_verify($password, $newUser['password_hash'])) {
            echo "✓ Password verification: SUCCESS\n\n";
        } else {
            echo "✗ Password verification: FAILED\n\n";
        }
        
    } else {
        echo "✓ User found:\n";
        echo "   ID: {$user['id']}\n";
        echo "   Name: {$user['name']}\n";
        echo "   Email: {$user['email']}\n";
        echo "   Role: {$user['role']}\n";
        echo "   Hospital ID: " . ($user['hospital_id'] ?? 'NULL') . "\n\n";
        
        // Check role
        if ($user['role'] !== 'super_admin') {
            echo "⚠ WARNING: Role is '{$user['role']}' but should be 'super_admin'\n";
            echo "Updating role...\n";
            $stmt = $pdo->prepare("UPDATE admins SET role = 'super_admin' WHERE id = ?");
            $stmt->execute([$user['id']]);
            echo "✓ Role updated to 'super_admin'\n\n";
        }
        
        // Test password
        echo "Step 2: Testing password...\n";
        if (password_verify($password, $user['password_hash'])) {
            echo "✓ Password verification: SUCCESS\n";
            echo "   Current password hash works correctly!\n\n";
        } else {
            echo "✗ Password verification: FAILED\n";
            echo "   Current hash doesn't match. Updating password...\n\n";
            
            // Update password with fresh hash
            $newPasswordHash = password_hash($password, PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("UPDATE admins SET password_hash = ? WHERE id = ?");
            $stmt->execute([$newPasswordHash, $user['id']]);
            
            echo "✓ Password updated!\n";
            
            // Verify again
            $stmt = $pdo->prepare("SELECT password_hash FROM admins WHERE id = ?");
            $stmt->execute([$user['id']]);
            $updatedUser = $stmt->fetch();
            
            if (password_verify($password, $updatedUser['password_hash'])) {
                echo "✓ New password verification: SUCCESS\n\n";
            } else {
                echo "✗ New password verification: FAILED\n\n";
            }
        }
    }
    
    // Final verification
    echo "Step 3: Final verification...\n";
    $stmt = $pdo->prepare("SELECT id, email, role, password_hash FROM admins WHERE email = ?");
    $stmt->execute([$email]);
    $finalUser = $stmt->fetch();
    
    if ($finalUser && $finalUser['role'] === 'super_admin' && password_verify($password, $finalUser['password_hash'])) {
        echo "✓ All checks passed!\n";
        echo "   User ID: {$finalUser['id']}\n";
        echo "   Email: {$finalUser['email']}\n";
        echo "   Role: {$finalUser['role']}\n";
        echo "   Password: Verified\n\n";
    } else {
        echo "✗ Verification failed!\n\n";
    }
    
    echo "========================================\n";
    echo "Login Information:\n";
    echo "========================================\n";
    echo "URL: http://localhost/hms_project/superadmin/login\n";
    echo "Email: {$email}\n";
    echo "Password: {$password}\n";
    echo "\n";
    echo "If login still fails, check:\n";
    echo "1. Browser cache clear karein\n";
    echo "2. Incognito/Private window mein try karein\n";
    echo "3. Error logs check karein (XAMPP error log)\n";
    
} catch (PDOException $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Database Error!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    echo "\n";
    echo "Please check:\n";
    echo "1. MySQL is running in XAMPP\n";
    echo "2. Database '{$dbName}' exists\n";
    echo "3. Database credentials are correct\n";
    exit(1);
} catch (Exception $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Error occurred!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}

echo "</pre>";
?>

