<?php
/**
 * Quick Fix - Create Missing Tables
 * Directly creates hospitals table and other missing tables
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<!DOCTYPE html><html><head><title>Quick Database Fix</title>";
echo "<style>body{font-family:Arial;padding:20px;background:#f5f5f5;}pre{background:#fff;padding:15px;border-radius:5px;box-shadow:0 2px 5px rgba(0,0,0,0.1);}</style>";
echo "</head><body>";
echo "<h2>Quick Database Fix</h2>";
echo "<pre>";

// Load database config
$dbConfigFile = __DIR__ . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'database.php';
if (!file_exists($dbConfigFile)) {
    die("✗ Database config file not found\n");
}

$dbConfig = require $dbConfigFile;

// Extract connection details
$host = '127.0.0.1';
$port = 3306;
$dbname = 'hms';
$user = $dbConfig['user'] ?? 'root';
$pass = $dbConfig['pass'] ?? '';

if (isset($dbConfig['dsn'])) {
    if (preg_match('/host=([^;]+)/', $dbConfig['dsn'], $matches)) {
        $host = $matches[1];
    }
    if (preg_match('/port=(\d+)/', $dbConfig['dsn'], $matches)) {
        $port = (int)$matches[1];
    }
    if (preg_match('/dbname=([^;]+)/', $dbConfig['dsn'], $matches)) {
        $dbname = $matches[1];
    }
}

try {
    // Connect to MySQL
    echo "Connecting to database...\n";
    $pdo = new PDO(
        "mysql:host={$host};port={$port};dbname={$dbname};charset=utf8mb4",
        $user,
        $pass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    echo "✓ Connected to database '{$dbname}'\n\n";
    
    // Check if hospitals table exists
    echo "Checking hospitals table...\n";
    $stmt = $pdo->query("SHOW TABLES LIKE 'hospitals'");
    $exists = $stmt->fetch() !== false;
    
    if (!$exists) {
        echo "✗ Hospitals table NOT found. Creating...\n\n";
        
        // Create hospitals table
        $sql = "CREATE TABLE IF NOT EXISTS `hospitals` (
          `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
          `name` VARCHAR(200) NOT NULL,
          `code` VARCHAR(50) NULL,
          `address` TEXT NULL,
          `city` VARCHAR(100) NULL,
          `state` VARCHAR(100) NULL,
          `country` VARCHAR(100) NULL DEFAULT 'India',
          `pincode` VARCHAR(20) NULL,
          `phone` VARCHAR(20) NULL,
          `email` VARCHAR(150) NULL,
          `website` VARCHAR(255) NULL,
          `registration_number` VARCHAR(100) NULL,
          `license_number` VARCHAR(100) NULL,
          `logo` VARCHAR(255) NULL,
          `status` ENUM('Active','Inactive','Suspended') NOT NULL DEFAULT 'Active',
          `subscription_plan` VARCHAR(50) NULL DEFAULT 'Basic',
          `subscription_expires_at` DATETIME NULL,
          `max_users` INT UNSIGNED NULL DEFAULT 10,
          `max_patients` INT UNSIGNED NULL,
          `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
          `updated_at` DATETIME NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
          PRIMARY KEY (`id`),
          UNIQUE KEY `uq_hospitals_code` (`code`),
          KEY `idx_hospitals_status` (`status`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
        
        $pdo->exec($sql);
        echo "✓ Hospitals table created successfully!\n\n";
    } else {
        echo "✓ Hospitals table already exists\n\n";
    }
    
    // Check and add hospital_id to admins table if needed
    echo "Checking admins table structure...\n";
    $stmt = $pdo->query("SHOW COLUMNS FROM admins LIKE 'hospital_id'");
    $hasHospitalId = $stmt->fetch() !== false;
    
    if (!$hasHospitalId) {
        echo "✗ hospital_id column NOT found in admins table. Adding...\n";
        try {
            $pdo->exec("ALTER TABLE admins ADD COLUMN hospital_id INT UNSIGNED NULL AFTER role");
            $pdo->exec("ALTER TABLE admins ADD KEY idx_admins_hospital (hospital_id)");
            // Add foreign key only if hospitals table exists
            if ($exists || $pdo->query("SHOW TABLES LIKE 'hospitals'")->fetch()) {
                try {
                    $pdo->exec("ALTER TABLE admins ADD CONSTRAINT fk_admins_hospital FOREIGN KEY (hospital_id) REFERENCES hospitals (id) ON DELETE SET NULL");
                } catch (PDOException $e) {
                    // Foreign key might already exist or fail, continue
                }
            }
            echo "✓ hospital_id column added to admins table\n\n";
        } catch (PDOException $e) {
            echo "⚠ Could not add hospital_id: " . $e->getMessage() . "\n\n";
        }
    } else {
        echo "✓ hospital_id column already exists in admins table\n\n";
    }
    
    // Check super admin user
    echo "Checking super admin user...\n";
    $stmt = $pdo->prepare("SELECT id, name, email, role FROM admins WHERE email = ? AND role = 'super_admin'");
    $stmt->execute(['superadmin@hms.local']);
    $superAdmin = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$superAdmin) {
        echo "✗ Super admin user NOT found. Creating...\n";
        $password = 'password';
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        
        $insertFields = ['name', 'email', 'role', 'password_hash', 'created_at'];
        $insertValues = ['Super Administrator', 'superadmin@hms.local', 'super_admin', $passwordHash, date('Y-m-d H:i:s')];
        
        if ($hasHospitalId || $pdo->query("SHOW COLUMNS FROM admins LIKE 'hospital_id'")->fetch()) {
            $insertFields[] = 'hospital_id';
            $insertValues[] = null;
        }
        
        $placeholders = implode(', ', array_fill(0, count($insertFields), '?'));
        $sql = "INSERT INTO admins (" . implode(', ', $insertFields) . ") VALUES ({$placeholders})";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($insertValues);
        
        echo "✓ Super admin user created\n";
        echo "   Email: superadmin@hms.local\n";
        echo "   Password: password\n\n";
    } else {
        echo "✓ Super admin user exists\n";
        echo "   ID: {$superAdmin['id']}\n";
        echo "   Email: {$superAdmin['email']}\n\n";
    }
    
    // Verify tables
    echo "Verifying tables...\n";
    $requiredTables = ['hospitals', 'admins', 'patients', 'doctors', 'departments'];
    $stmt = $pdo->query("SHOW TABLES");
    $existingTables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    foreach ($requiredTables as $table) {
        if (in_array($table, $existingTables)) {
            echo "  ✓ {$table}\n";
        } else {
            echo "  ✗ {$table} (MISSING)\n";
        }
    }
    
    echo "\n";
    echo "========================================\n";
    echo "✓ Quick Fix Complete!\n";
    echo "========================================\n";
    echo "\n";
    echo "You can now:\n";
    echo "1. Access Super Admin: http://localhost/hms_project/superadmin/login\n";
    echo "2. Login with: superadmin@hms.local / password\n";
    echo "\n";
    echo "To create ALL tables, run: setup_database_mvc.php\n";
    
} catch (PDOException $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Error!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    echo "\n";
    echo "Please check:\n";
    echo "1. MySQL server is running\n";
    echo "2. Database '{$dbname}' exists\n";
    echo "3. Database credentials are correct\n";
}

echo "</pre>";
echo "</body></html>";
?>

