<?php
/**
 * Database Setup Script for HMS Project
 * Run this file once to create the database and tables
 * 
 * Usage: php setup_database.php
 * Or access via browser: http://localhost/hms_project/setup_database.php
 */

error_reporting(E_ALL);
ini_set('display_errors', '1');

// Database configuration
$dbHost = '127.0.0.1';
$dbPort = '3306';
$dbUser = 'root';
$dbPass = '';
$dbName = 'hms';

echo "<h2>HMS Database Setup</h2>";
echo "<pre>";

try {
    // Step 1: Connect to MySQL without selecting database
    echo "Step 1: Connecting to MySQL server...\n";
    $pdo = new PDO(
        "mysql:host={$dbHost};port={$dbPort};charset=utf8mb4",
        $dbUser,
        $dbPass,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );
    echo "✓ Connected successfully\n\n";

    // Step 2: Create database if not exists
    echo "Step 2: Creating database '{$dbName}'...\n";
    $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$dbName}` CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci");
    echo "✓ Database created/verified\n\n";

    // Step 3: Select the database
    echo "Step 3: Selecting database...\n";
    $pdo->exec("USE `{$dbName}`");
    echo "✓ Database selected\n\n";

    // Step 4: Read and execute SQL file
    echo "Step 4: Reading database.sql file...\n";
    $sqlFile = __DIR__ . DIRECTORY_SEPARATOR . 'database.sql';
    
    if (!file_exists($sqlFile)) {
        throw new RuntimeException("SQL file not found: {$sqlFile}");
    }
    
    $sql = file_get_contents($sqlFile);
    
    // Remove CREATE DATABASE and USE statements (already handled)
    $sql = preg_replace('/CREATE DATABASE.*?;/i', '', $sql);
    $sql = preg_replace('/USE\s+`?hms`?;?/i', '', $sql);
    
    echo "✓ SQL file loaded\n\n";

    // Step 5: Execute SQL statements
    echo "Step 5: Creating tables...\n";
    
    // Better SQL parsing - handle multi-line statements
    $statements = [];
    $currentStatement = '';
    $inString = false;
    $stringChar = '';
    
    $lines = explode("\n", $sql);
    foreach ($lines as $line) {
        $line = trim($line);
        
        // Skip comments
        if (empty($line) || preg_match('/^--/', $line)) {
            continue;
        }
        
        // Track string literals to avoid splitting on semicolons inside strings
        $chars = str_split($line);
        foreach ($chars as $char) {
            if (($char === '"' || $char === "'" || $char === '`') && !$inString) {
                $inString = true;
                $stringChar = $char;
            } elseif ($char === $stringChar && $inString) {
                $inString = false;
                $stringChar = '';
            }
            
            $currentStatement .= $char;
            
            if ($char === ';' && !$inString) {
                $stmt = trim($currentStatement);
                if (!empty($stmt) && strlen($stmt) > 1) {
                    $statements[] = $stmt;
                }
                $currentStatement = '';
            }
        }
        
        // Add newline if not at end of statement
        if (!empty($currentStatement)) {
            $currentStatement .= "\n";
        }
    }
    
    // Add last statement if no semicolon at end
    if (!empty(trim($currentStatement))) {
        $statements[] = trim($currentStatement);
    }
    
    $tableCount = 0;
    foreach ($statements as $statement) {
        if (empty(trim($statement))) continue;
        try {
            $pdo->exec($statement);
            if (preg_match('/CREATE\s+TABLE.*?`(\w+)`/i', $statement, $matches)) {
                $tableCount++;
                echo "  ✓ Created table: {$matches[1]}\n";
            }
        } catch (PDOException $e) {
            // Ignore "table already exists" errors
            if (strpos($e->getMessage(), 'already exists') !== false) {
                if (preg_match('/CREATE\s+TABLE.*?`(\w+)`/i', $statement, $matches)) {
                    echo "  ℹ Table already exists: {$matches[1]}\n";
                }
            } else {
                echo "  ⚠ Error: " . $e->getMessage() . "\n";
                // Show first 200 chars of failed statement for debugging
                $stmtPreview = substr($statement, 0, 200);
                echo "     Statement: " . $stmtPreview . "...\n";
            }
        }
    }
    
    echo "\n✓ Total tables processed: {$tableCount}\n\n";

    // Step 6: Create default admin users
    echo "Step 6: Creating default admin users...\n";
    
    // Create super admin
    $superAdminEmail = 'superadmin@hms.local';
    $superAdminPassword = 'password';
    $superAdminName = 'Super Administrator';
    
    $stmt = $pdo->prepare("SELECT id FROM admins WHERE email = ?");
    $stmt->execute([$superAdminEmail]);
    $existingSuperAdmin = $stmt->fetch();
    
    if ($existingSuperAdmin) {
        echo "  ✓ Super admin user already exists\n";
    } else {
        $passwordHash = password_hash($superAdminPassword, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("
            INSERT INTO admins (name, email, role, hospital_id, password_hash) 
            VALUES (?, ?, 'super_admin', NULL, ?)
        ");
        $stmt->execute([$superAdminName, $superAdminEmail, $passwordHash]);
        echo "  ✓ Super admin user created\n";
        echo "     Email: {$superAdminEmail}\n";
        echo "     Password: {$superAdminPassword}\n";
    }
    
    // Create regular admin
    $adminEmail = 'admin@local';
    $adminPassword = 'admin123';
    $adminName = 'Administrator';
    
    $stmt = $pdo->prepare("SELECT id FROM admins WHERE email = ?");
    $stmt->execute([$adminEmail]);
    $existingAdmin = $stmt->fetch();
    
    if ($existingAdmin) {
        echo "  ✓ Admin user already exists\n";
    } else {
        $passwordHash = password_hash($adminPassword, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("
            INSERT INTO admins (name, email, role, hospital_id, password_hash) 
            VALUES (?, ?, 'admin', NULL, ?)
        ");
        $stmt->execute([$adminName, $adminEmail, $passwordHash]);
        echo "  ✓ Default admin user created\n";
        echo "     Email: {$adminEmail}\n";
        echo "     Password: {$adminPassword}\n";
    }
    
    echo "\n";
    echo "========================================\n";
    echo "✓ Database setup completed successfully!\n";
    echo "========================================\n";
    echo "\n";
    echo "You can now:\n";
    echo "1. Delete this setup_database.php file for security\n";
    echo "2. Access Super Admin Panel: http://localhost/hms_project/superadmin/login\n";
    echo "   Login: superadmin@hms.local / password\n";
    echo "3. Access Regular Admin Panel: http://localhost/hms_project/admin/login\n";
    echo "   Login: admin@local / admin123\n";
    
} catch (PDOException $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Database setup failed!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    echo "\n";
    echo "Please check:\n";
    echo "1. MySQL server is running\n";
    echo "2. Database credentials are correct in setup_database.php\n";
    echo "3. User has permission to create databases\n";
    exit(1);
} catch (Exception $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Setup failed!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}

echo "</pre>";
?>

