<?php
/**
 * Complete Database Setup Script
 * This will create all tables including hospitals table
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h2>Complete Database Setup</h2>";
echo "<pre>";

// Database configuration
$dbConfig = require __DIR__ . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'database.php';

$host = '127.0.0.1';
$port = 3306;
$dbname = 'hms';
$user = $dbConfig['user'] ?? 'root';
$pass = $dbConfig['pass'] ?? '';

// Extract from DSN if available
if (isset($dbConfig['dsn'])) {
    if (preg_match('/host=([^;]+)/', $dbConfig['dsn'], $matches)) {
        $host = $matches[1];
    }
    if (preg_match('/port=(\d+)/', $dbConfig['dsn'], $matches)) {
        $port = (int)$matches[1];
    }
    if (preg_match('/dbname=([^;]+)/', $dbConfig['dsn'], $matches)) {
        $dbname = $matches[1];
    }
}

try {
    // Connect to MySQL (without database)
    $pdo = new PDO(
        "mysql:host={$host};port={$port};charset=utf8mb4",
        $user,
        $pass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    
    echo "Step 1: Creating database if not exists...\n";
    $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$dbname}` CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci");
    echo "✓ Database '{$dbname}' ready\n";
    
    // Connect to the database
    $pdo = new PDO(
        "mysql:host={$host};port={$port};dbname={$dbname};charset=utf8mb4",
        $user,
        $pass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    
    echo "\nStep 2: Reading database.sql file...\n";
    $sqlFile = __DIR__ . DIRECTORY_SEPARATOR . 'database.sql';
    if (!file_exists($sqlFile)) {
        throw new Exception("database.sql file not found: {$sqlFile}");
    }
    
    $sql = file_get_contents($sqlFile);
    if ($sql === false) {
        throw new Exception("Could not read database.sql file");
    }
    
    echo "✓ SQL file loaded (" . number_format(strlen($sql)) . " bytes)\n";
    
    echo "\nStep 3: Executing SQL statements...\n";
    
    // Remove comments and split by semicolon
    $sql = preg_replace('/--.*$/m', '', $sql);
    $statements = array_filter(
        array_map('trim', explode(';', $sql)),
        function($stmt) {
            return !empty($stmt) && !preg_match('/^(USE|CREATE DATABASE)/i', $stmt);
        }
    ));
    
    $executed = 0;
    $errors = [];
    
    foreach ($statements as $statement) {
        $statement = trim($statement);
        if (empty($statement)) continue;
        
        try {
            $pdo->exec($statement);
            $executed++;
        } catch (PDOException $e) {
            // Ignore "already exists" errors
            if (strpos($e->getMessage(), 'already exists') === false && 
                strpos($e->getMessage(), 'Duplicate') === false) {
                $errors[] = $e->getMessage();
            }
        }
    }
    
    echo "✓ Executed {$executed} SQL statements\n";
    if (!empty($errors)) {
        echo "⚠ Some errors (may be expected):\n";
        foreach (array_slice($errors, 0, 5) as $error) {
            echo "   - " . substr($error, 0, 100) . "\n";
        }
        if (count($errors) > 5) {
            echo "   ... and " . (count($errors) - 5) . " more\n";
        }
    }
    
    echo "\nStep 4: Verifying tables...\n";
    $requiredTables = [
        'hospitals',
        'admins',
        'patients',
        'doctors',
        'departments',
        'appointments',
    ];
    
    $stmt = $pdo->query("SHOW TABLES");
    $existingTables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $missing = [];
    foreach ($requiredTables as $table) {
        if (in_array($table, $existingTables)) {
            echo "✓ Table '{$table}' exists\n";
        } else {
            echo "✗ Table '{$table}' NOT found\n";
            $missing[] = $table;
        }
    }
    
    if (!empty($missing)) {
        echo "\n⚠ Warning: Some tables are missing. Database may not be fully set up.\n";
    }
    
    echo "\nStep 5: Checking super admin user...\n";
    $stmt = $pdo->prepare("SELECT id, name, email, role FROM admins WHERE email = ? AND role = 'super_admin'");
    $stmt->execute(['superadmin@hms.local']);
    $superAdmin = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($superAdmin) {
        echo "✓ Super admin user exists\n";
        echo "   ID: {$superAdmin['id']}\n";
        echo "   Email: {$superAdmin['email']}\n";
        echo "   Role: {$superAdmin['role']}\n";
    } else {
        echo "⚠ Super admin user not found. Creating...\n";
        $password = 'password';
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        
        try {
            // Check if hospital_id column exists
            $stmt = $pdo->query("SHOW COLUMNS FROM admins LIKE 'hospital_id'");
            $hasHospitalId = $stmt->fetch() !== false;
            
            if ($hasHospitalId) {
                $stmt = $pdo->prepare("
                    INSERT INTO admins (name, email, role, hospital_id, password_hash, created_at) 
                    VALUES (?, ?, 'super_admin', NULL, ?, NOW())
                ");
                $stmt->execute(['Super Administrator', 'superadmin@hms.local', $passwordHash]);
            } else {
                $stmt = $pdo->prepare("
                    INSERT INTO admins (name, email, role, password_hash, created_at) 
                    VALUES (?, ?, 'super_admin', ?, NOW())
                ");
                $stmt->execute(['Super Administrator', 'superadmin@hms.local', $passwordHash]);
            }
            
            echo "✓ Super admin user created\n";
            echo "   Email: superadmin@hms.local\n";
            echo "   Password: password\n";
        } catch (Exception $e) {
            echo "✗ Failed to create super admin: " . $e->getMessage() . "\n";
        }
    }
    
    echo "\n";
    echo "========================================\n";
    echo "✓ Database setup completed!\n";
    echo "========================================\n";
    echo "\n";
    echo "You can now:\n";
    echo "1. Access Super Admin Panel: http://localhost/hms_project/superadmin/login\n";
    echo "2. Login with: superadmin@hms.local / password\n";
    echo "\n";
    
} catch (Exception $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Setup failed!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    echo "\n";
    echo "Please check:\n";
    echo "1. MySQL server is running\n";
    echo "2. Database credentials are correct\n";
    echo "3. User has permission to create databases\n";
}

echo "</pre>";
?>

