<?php
/**
 * Complete Database Setup - MVC Pattern
 * Creates all missing database tables according to MVC structure
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<!DOCTYPE html><html><head><title>Database Setup</title>";
echo "<style>body{font-family:Arial;padding:20px;background:#f5f5f5;}pre{background:#fff;padding:15px;border-radius:5px;box-shadow:0 2px 5px rgba(0,0,0,0.1);}</style>";
echo "</head><body>";
echo "<h2>Database Setup - MVC Pattern</h2>";
echo "<pre>";

// Load database config
$dbConfigFile = __DIR__ . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'database.php';
if (!file_exists($dbConfigFile)) {
    die("✗ Database config file not found: {$dbConfigFile}\n");
}

$dbConfig = require $dbConfigFile;

// Extract connection details
$host = '127.0.0.1';
$port = 3306;
$dbname = 'hms';
$user = $dbConfig['user'] ?? 'root';
$pass = $dbConfig['pass'] ?? '';

if (isset($dbConfig['dsn'])) {
    if (preg_match('/host=([^;]+)/', $dbConfig['dsn'], $matches)) {
        $host = $matches[1];
    }
    if (preg_match('/port=(\d+)/', $dbConfig['dsn'], $matches)) {
        $port = (int)$matches[1];
    }
    if (preg_match('/dbname=([^;]+)/', $dbConfig['dsn'], $matches)) {
        $dbname = $matches[1];
    }
}

try {
    // Step 1: Connect to MySQL and create database
    echo "Step 1: Creating database...\n";
    $pdo = new PDO(
        "mysql:host={$host};port={$port};charset=utf8mb4",
        $user,
        $pass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    
    $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$dbname}` CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci");
    echo "✓ Database '{$dbname}' created/verified\n";
    
    // Connect to the database
    $pdo = new PDO(
        "mysql:host={$host};port={$port};dbname={$dbname};charset=utf8mb4",
        $user,
        $pass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    
    // Step 2: Check existing tables
    echo "\nStep 2: Checking existing tables...\n";
    $stmt = $pdo->query("SHOW TABLES");
    $existingTables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    echo "Found " . count($existingTables) . " existing table(s)\n";
    
    // Step 3: Read and execute database.sql
    echo "\nStep 3: Reading database.sql file...\n";
    $sqlFile = __DIR__ . DIRECTORY_SEPARATOR . 'database.sql';
    if (!file_exists($sqlFile)) {
        throw new Exception("database.sql file not found: {$sqlFile}");
    }
    
    $sql = file_get_contents($sqlFile);
    if ($sql === false) {
        throw new Exception("Could not read database.sql file");
    }
    
    echo "✓ SQL file loaded (" . number_format(strlen($sql)) . " bytes)\n";
    
    // Step 4: Execute SQL statements
    echo "\nStep 4: Creating/updating tables...\n";
    
    // Remove comments and split by semicolon
    $sql = preg_replace('/--.*$/m', '', $sql);
    $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);
    
    // Split by semicolon but keep CREATE TABLE statements together
    $statements = [];
    $current = '';
    $inCreateTable = false;
    
    $lines = explode("\n", $sql);
    foreach ($lines as $line) {
        $line = trim($line);
        if (empty($line)) continue;
        
        if (preg_match('/^CREATE\s+TABLE/i', $line)) {
            $inCreateTable = true;
            $current = $line;
        } elseif ($inCreateTable) {
            $current .= "\n" . $line;
            if (preg_match('/\);?\s*$/i', $line)) {
                $inCreateTable = false;
                $statements[] = trim($current);
                $current = '';
            }
        } elseif (preg_match('/^ALTER\s+TABLE/i', $line)) {
            $statements[] = $line;
        } elseif (preg_match('/^INSERT\s+IGNORE/i', $line)) {
            $statements[] = $line;
        }
    }
    
    if (!empty($current)) {
        $statements[] = trim($current);
    }
    
    $created = 0;
    $updated = 0;
    $skipped = 0;
    $errors = [];
    
    foreach ($statements as $statement) {
        $statement = trim($statement);
        if (empty($statement)) continue;
        
        try {
            // Check if it's a CREATE TABLE statement
            if (preg_match('/CREATE\s+TABLE\s+(?:IF\s+NOT\s+EXISTS\s+)?`?(\w+)`?/i', $statement, $matches)) {
                $tableName = $matches[1];
                if (in_array($tableName, $existingTables)) {
                    echo "  ⊙ Table '{$tableName}' already exists, skipping...\n";
                    $skipped++;
                    continue;
                }
            }
            
            $pdo->exec($statement);
            
            if (preg_match('/CREATE\s+TABLE/i', $statement)) {
                $created++;
                if (preg_match('/CREATE\s+TABLE\s+(?:IF\s+NOT\s+EXISTS\s+)?`?(\w+)`?/i', $statement, $matches)) {
                    echo "  ✓ Created table '{$matches[1]}'\n";
                }
            } elseif (preg_match('/ALTER\s+TABLE/i', $statement)) {
                $updated++;
            } elseif (preg_match('/INSERT/i', $statement)) {
                // Insert statement executed
            }
        } catch (PDOException $e) {
            // Ignore "already exists" and "duplicate" errors
            if (strpos($e->getMessage(), 'already exists') === false && 
                strpos($e->getMessage(), 'Duplicate') === false &&
                strpos($e->getMessage(), 'Duplicate key') === false) {
                $errors[] = [
                    'message' => $e->getMessage(),
                    'code' => $e->getCode()
                ];
            } else {
                $skipped++;
            }
        }
    }
    
    echo "\n✓ Created {$created} new table(s)\n";
    if ($updated > 0) {
        echo "✓ Updated {$updated} table(s)\n";
    }
    if ($skipped > 0) {
        echo "⊙ Skipped {$skipped} statement(s) (already exists)\n";
    }
    
    // Step 5: Verify all required tables
    echo "\nStep 5: Verifying required tables...\n";
    $requiredTables = [
        'hospitals' => 'Multi-hospital support',
        'admins' => 'Admin users',
        'patients' => 'Patient records',
        'doctors' => 'Doctor records',
        'departments' => 'Hospital departments',
        'appointments' => 'Patient appointments',
        'rooms' => 'Hospital rooms',
        'beds' => 'Hospital beds',
        'admissions' => 'IPD admissions',
        'invoices' => 'Billing invoices',
        'invoice_items' => 'Invoice line items',
        'payments' => 'Payment records',
        'medicines' => 'Pharmacy medicines',
        'stock_movements' => 'Stock movements',
        'pharmacy_sales' => 'Pharmacy sales',
        'lab_tests' => 'Laboratory tests',
        'lab_orders' => 'Lab test orders',
        'lab_results' => 'Lab test results',
        'hr_staff' => 'HR staff records',
        'services' => 'Hospital services',
        'charges' => 'Service charges',
        'doctor_schedules' => 'Doctor schedules',
    ];
    
    $stmt = $pdo->query("SHOW TABLES");
    $existingTables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $missing = [];
    $present = [];
    
    foreach ($requiredTables as $table => $description) {
        if (in_array($table, $existingTables)) {
            echo "  ✓ {$table} - {$description}\n";
            $present[] = $table;
        } else {
            echo "  ✗ {$table} - {$description} (MISSING)\n";
            $missing[] = $table;
        }
    }
    
    // Step 6: Check and create super admin user
    echo "\nStep 6: Checking super admin user...\n";
    
    // Check if hospital_id column exists in admins table
    $stmt = $pdo->query("SHOW COLUMNS FROM admins LIKE 'hospital_id'");
    $hasHospitalId = $stmt->fetch() !== false;
    
    $stmt = $pdo->prepare("SELECT id, name, email, role FROM admins WHERE email = ? AND role = 'super_admin'");
    $stmt->execute(['superadmin@hms.local']);
    $superAdmin = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($superAdmin) {
        echo "✓ Super admin user exists\n";
        echo "   ID: {$superAdmin['id']}\n";
        echo "   Name: {$superAdmin['name']}\n";
        echo "   Email: {$superAdmin['email']}\n";
        
        // Verify password
        $stmt = $pdo->prepare("SELECT password_hash FROM admins WHERE id = ?");
        $stmt->execute([$superAdmin['id']]);
        $hash = $stmt->fetchColumn();
        
        if ($hash && password_verify('password', $hash)) {
            echo "   ✓ Password verified\n";
        } else {
            echo "   ⚠ Updating password...\n";
            $newHash = password_hash('password', PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("UPDATE admins SET password_hash = ? WHERE id = ?");
            $stmt->execute([$newHash, $superAdmin['id']]);
            echo "   ✓ Password updated\n";
        }
    } else {
        echo "⚠ Super admin user not found. Creating...\n";
        $password = 'password';
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        
        if ($hasHospitalId) {
            $stmt = $pdo->prepare("
                INSERT INTO admins (name, email, role, hospital_id, password_hash, created_at) 
                VALUES (?, ?, 'super_admin', NULL, ?, NOW())
            ");
            $stmt->execute(['Super Administrator', 'superadmin@hms.local', $passwordHash]);
        } else {
            $stmt = $pdo->prepare("
                INSERT INTO admins (name, email, role, password_hash, created_at) 
                VALUES (?, ?, 'super_admin', ?, NOW())
            ");
            $stmt->execute(['Super Administrator', 'superadmin@hms.local', $passwordHash]);
        }
        
        echo "✓ Super admin user created\n";
        echo "   Email: superadmin@hms.local\n";
        echo "   Password: password\n";
    }
    
    // Step 7: Summary
    echo "\n";
    echo "========================================\n";
    echo "Database Setup Summary\n";
    echo "========================================\n";
    echo "Total tables required: " . count($requiredTables) . "\n";
    echo "Tables present: " . count($present) . "\n";
    if (!empty($missing)) {
        echo "Tables missing: " . count($missing) . "\n";
        echo "\nMissing tables:\n";
        foreach ($missing as $table) {
            echo "  - {$table}\n";
        }
    } else {
        echo "✓ All required tables are present!\n";
    }
    
    echo "\n";
    echo "========================================\n";
    echo "✓ Setup Complete!\n";
    echo "========================================\n";
    echo "\n";
    echo "Next Steps:\n";
    echo "1. Access Super Admin Panel: http://localhost/hms_project/superadmin/login\n";
    echo "2. Login with: superadmin@hms.local / password\n";
    echo "3. Create hospitals and manage the system\n";
    echo "\n";
    
} catch (Exception $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Setup Failed!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    echo "\n";
    echo "Please check:\n";
    echo "1. MySQL server is running\n";
    echo "2. Database credentials in: admin/app/config/database.php\n";
    echo "3. User has permission to create databases and tables\n";
    echo "\n";
    echo "Stack trace:\n";
    echo $e->getTraceAsString() . "\n";
}

echo "</pre>";
echo "</body></html>";
?>

