<?php
/**
 * Multi-Hospital Database Setup Script
 * Run this file once to add multi-hospital support to existing database
 * 
 * Usage: php setup_multi_hospital.php
 * Or access via browser: http://localhost/hms_project/setup_multi_hospital.php
 */

error_reporting(E_ALL);
ini_set('display_errors', '1');

// Database configuration
$dbHost = '127.0.0.1';
$dbPort = '3306';
$dbUser = 'root';
$dbPass = '';
$dbName = 'hms';

echo "<h2>Multi-Hospital Database Setup</h2>";
echo "<pre>";

try {
    // Step 1: Connect to MySQL
    echo "Step 1: Connecting to MySQL server...\n";
    $pdo = new PDO(
        "mysql:host={$dbHost};port={$dbPort};dbname={$dbName};charset=utf8mb4",
        $dbUser,
        $dbPass,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );
    echo "✓ Connected successfully\n\n";

    // Step 2: Read and execute SQL file
    echo "Step 2: Reading database_multi_hospital.sql file...\n";
    $sqlFile = __DIR__ . DIRECTORY_SEPARATOR . 'database_multi_hospital.sql';
    
    if (!file_exists($sqlFile)) {
        throw new RuntimeException("SQL file not found: {$sqlFile}");
    }
    
    $sql = file_get_contents($sqlFile);
    
    // Remove USE statement (already connected)
    $sql = preg_replace('/USE\s+`?hms`?;?/i', '', $sql);
    
    echo "✓ SQL file loaded\n\n";

    // Step 3: Execute SQL statements
    echo "Step 3: Creating/updating tables...\n";
    
    // Split by semicolon but handle multi-line statements
    $statements = [];
    $currentStatement = '';
    $inString = false;
    $stringChar = '';
    
    $lines = explode("\n", $sql);
    foreach ($lines as $line) {
        $line = trim($line);
        
        // Skip comments and empty lines
        if (empty($line) || preg_match('/^--/', $line)) {
            continue;
        }
        
        // Track string literals
        $chars = str_split($line);
        foreach ($chars as $char) {
            if (($char === '"' || $char === "'" || $char === '`') && !$inString) {
                $inString = true;
                $stringChar = $char;
            } elseif ($char === $stringChar && $inString) {
                $inString = false;
                $stringChar = '';
            }
            
            $currentStatement .= $char;
            
            if ($char === ';' && !$inString) {
                $stmt = trim($currentStatement);
                if (!empty($stmt) && strlen($stmt) > 1) {
                    $statements[] = $stmt;
                }
                $currentStatement = '';
            }
        }
        
        if (!empty($currentStatement)) {
            $currentStatement .= "\n";
        }
    }
    
    if (!empty(trim($currentStatement))) {
        $statements[] = trim($currentStatement);
    }
    
    $successCount = 0;
    $errorCount = 0;
    
    foreach ($statements as $statement) {
        if (empty(trim($statement))) continue;
        try {
            $pdo->exec($statement);
            $successCount++;
            
            // Extract table/operation name for logging
            if (preg_match('/CREATE\s+TABLE.*?`(\w+)`/i', $statement, $matches)) {
                echo "  ✓ Created table: {$matches[1]}\n";
            } elseif (preg_match('/ALTER\s+TABLE.*?`(\w+)`/i', $statement, $matches)) {
                echo "  ✓ Updated table: {$matches[1]}\n";
            } elseif (preg_match('/INSERT.*?INTO.*?`(\w+)`/i', $statement, $matches)) {
                echo "  ✓ Inserted into: {$matches[1]}\n";
            }
        } catch (PDOException $e) {
            $errorCount++;
            // Ignore "already exists" or "duplicate column" errors
            if (strpos($e->getMessage(), 'already exists') !== false || 
                strpos($e->getMessage(), 'Duplicate column') !== false ||
                strpos($e->getMessage(), 'Duplicate key') !== false) {
                // Silent ignore for idempotent operations
            } else {
                echo "  ⚠ Error: " . $e->getMessage() . "\n";
                $stmtPreview = substr($statement, 0, 200);
                echo "     Statement: " . $stmtPreview . "...\n";
            }
        }
    }
    
    echo "\n✓ Total statements executed: {$successCount}\n";
    if ($errorCount > 0) {
        echo "⚠ Errors encountered: {$errorCount} (mostly expected for existing structures)\n";
    }
    
    // Step 4: Create default hospital (if none exists)
    echo "\nStep 4: Creating default hospital...\n";
    $stmt = $pdo->query("SELECT COUNT(*) FROM hospitals");
    $hospitalCount = (int)$stmt->fetchColumn();
    
    if ($hospitalCount === 0) {
        $stmt = $pdo->prepare("
            INSERT INTO hospitals (name, code, status, created_at) 
            VALUES ('Default Hospital', 'HOSP001', 'Active', NOW())
        ");
        $stmt->execute();
        echo "  ✓ Default hospital created\n";
    } else {
        echo "  ✓ Hospitals already exist ({$hospitalCount} found)\n";
    }
    
    echo "\n";
    echo "========================================\n";
    echo "✓ Multi-Hospital setup completed!\n";
    echo "========================================\n";
    echo "\n";
    echo "Next steps:\n";
    echo "1. Access Super Admin Panel: http://localhost/hms_project/superadmin/login\n";
    echo "2. Default Super Admin: superadmin@hms.local / password\n";
    echo "3. Create hospitals and assign admins\n";
    
} catch (PDOException $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Database setup failed!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
} catch (Exception $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Setup failed!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}

echo "</pre>";
?>

