<?php
declare(strict_types=1);

class SuperAdminController extends Controller
{
    public function dashboard(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        
        $hospitalModel = new Hospital();
        $adminModel = new Admin();
        
        $totalHospitals = $hospitalModel->count();
        $activeHospitals = $hospitalModel->count(['status' => 'Active']);
        $totalAdmins = (int)$adminModel->db()->query("SELECT COUNT(*) FROM admins WHERE role != 'super_admin'")->fetchColumn();
        
        $recentHospitals = $hospitalModel->list(['limit' => 5]);
        
        // Get hospital-wise statistics
        $hospitalStats = [];
        try {
            $allHospitals = $hospitalModel->all();
            foreach ($allHospitals as $hospital) {
                $stats = $hospitalModel->getStats((int)$hospital['id']);
                $hospitalStats[] = [
                    'hospital' => $hospital,
                    'stats' => $stats,
                ];
            }
        } catch (Throwable $e) {
            $hospitalStats = [];
        }
        
        // Overall statistics
        $overallStats = [
            'total_patients' => 0,
            'total_doctors' => 0,
            'total_appointments' => 0,
            'total_admins' => $totalAdmins,
        ];
        
        try {
            $overallStats['total_patients'] = (int)$adminModel->db()->query("SELECT COUNT(*) FROM patients")->fetchColumn();
            $overallStats['total_doctors'] = (int)$adminModel->db()->query("SELECT COUNT(*) FROM doctors")->fetchColumn();
            $overallStats['total_appointments'] = (int)$adminModel->db()->query("SELECT COUNT(*) FROM appointments")->fetchColumn();
        } catch (Throwable $e) {
            // Ignore errors
        }
        
        $this->view('dashboard/index', [
            'title' => 'Super Admin Dashboard',
            'base' => defined('BASE_URL') ? BASE_URL : '/hms_project/superadmin',
            'totalHospitals' => $totalHospitals,
            'activeHospitals' => $activeHospitals,
            'totalAdmins' => $totalAdmins,
            'recentHospitals' => $recentHospitals,
            'hospitalStats' => $hospitalStats,
            'overallStats' => $overallStats,
        ]);
    }

    public function login(): void
    {
        if (Auth::check() && (Auth::user()['role'] ?? '') === 'super_admin') {
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/');
            exit;
        }
        
        $this->view('auth/login', [
            'title' => 'Super Admin Login',
            'base' => defined('BASE_URL') ? BASE_URL : '/hms_project/superadmin',
            'error' => $_GET['error'] ?? null,
        ]);
    }

    public function doLogin(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $email = trim($_POST['email'] ?? '');
        $password = (string)($_POST['password'] ?? '');

        if (empty($email) || empty($password)) {
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/login?error=Email%20and%20password%20required');
            exit;
        }

        try {
            // Ensure Admin model is loaded
            if (!class_exists('Admin')) {
                require_once BASE_PATH . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'Admin.php';
            }
            
            $adminModel = new Admin();
            $user = $adminModel->findByEmail($email);

            if (!$user) {
                error_log("Login failed: User not found - {$email}");
                header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/login?error=Invalid%20credentials');
                exit;
            }

            if (!password_verify($password, (string)($user['password_hash'] ?? ''))) {
                error_log("Login failed: Invalid password for - {$email}");
                header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/login?error=Invalid%20credentials');
                exit;
            }

            // Check if user is super admin
            if (($user['role'] ?? '') !== 'super_admin') {
                error_log("Login failed: Not super admin - {$email}, role: " . ($user['role'] ?? 'none'));
                header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/login?error=Access%20denied.%20Only%20super%20admin%20can%20login%20here.');
                exit;
            }

            // Start session if not already started
            if (session_status() !== PHP_SESSION_ACTIVE) {
                session_start();
            }
            
            // Ensure Auth class is loaded
            if (!class_exists('Auth')) {
                require_once BASE_PATH . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'Auth.php';
            }
            
            Auth::login($user);
            
            // Save session before redirect
            session_write_close();
            
            // Redirect to dashboard
            $baseUrl = defined('BASE_URL') ? BASE_URL : '/hms_project/superadmin';
            header('Location: ' . $baseUrl . '/');
            exit;
        } catch (Throwable $e) {
            error_log("Login error: " . $e->getMessage());
            error_log("Stack trace: " . $e->getTraceAsString());
            $errorMsg = "Login failed: " . $e->getMessage();
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/login?error=' . urlencode($errorMsg));
            exit;
        }
    }

    public function logout(): void
    {
        Auth::logout();
        session_unset();
        session_destroy();
        header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/login');
        exit;
    }

    public function hospitals(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        
        $q = trim((string)($_GET['q'] ?? ''));
        $status = $_GET['status'] ?? null;
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $model = new Hospital();
        $hospitals = $model->list(['q' => $q, 'status' => $status, 'limit' => $perPage, 'offset' => $offset]);
        $total = $model->count(['q' => $q, 'status' => $status]);

        // Get user count for each hospital
        $adminModel = new Admin();
        foreach ($hospitals as &$hospital) {
            try {
                $stmt = $adminModel->db()->prepare("SELECT COUNT(*) FROM admins WHERE hospital_id = ? AND role != 'super_admin'");
                $stmt->execute([(int)$hospital['id']]);
                $hospital['user_count'] = (int)$stmt->fetchColumn();
            } catch (Throwable $e) {
                $hospital['user_count'] = 0;
            }
        }
        unset($hospital);

        $flashSuccess = $_SESSION['flash_success'] ?? null;
        if (isset($_SESSION['flash_success'])) { unset($_SESSION['flash_success']); }

        $this->view('hospitals/list', [
            'title' => 'Hospitals',
            'hospitals' => $hospitals,
            'q' => $q,
            'status' => $status,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
            'flashSuccess' => $flashSuccess,
        ]);
    }

    public function addHospital(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        
        $formError = $_SESSION['form_error'] ?? null;
        $formErrors = $_SESSION['form_errors'] ?? [];
        $old = $_SESSION['old'] ?? null;
        if (isset($_SESSION['form_error'])) { unset($_SESSION['form_error']); }
        if (isset($_SESSION['form_errors'])) { unset($_SESSION['form_errors']); }
        if (isset($_SESSION['old'])) { unset($_SESSION['old']); }

        $this->view('hospitals/add', [
            'title' => 'Add Hospital',
            'formError' => $formError,
            'formErrors' => $formErrors,
            'old' => $old,
        ]);
    }

    public function storeHospital(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? ''),
            'address' => trim($_POST['address'] ?? ''),
            'city' => trim($_POST['city'] ?? ''),
            'state' => trim($_POST['state'] ?? ''),
            'country' => trim($_POST['country'] ?? 'India'),
            'pincode' => trim($_POST['pincode'] ?? ''),
            'phone' => trim($_POST['phone'] ?? ''),
            'email' => trim($_POST['email'] ?? ''),
            'website' => trim($_POST['website'] ?? ''),
            'registration_number' => trim($_POST['registration_number'] ?? ''),
            'license_number' => trim($_POST['license_number'] ?? ''),
            'status' => trim($_POST['status'] ?? 'Active'),
            'subscription_plan' => trim($_POST['subscription_plan'] ?? 'Basic'),
            'max_users' => isset($_POST['max_users']) ? (int)$_POST['max_users'] : 10,
            'max_patients' => isset($_POST['max_patients']) ? (int)$_POST['max_patients'] : null,
        ];

        $errors = $this->validateHospital($data);

        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/hospitals/add');
            exit;
        }

        try {
            $model = new Hospital();
            $id = $model->create($data);

            // Auto-create admin user for this hospital
            $defaultEmail = null;
            $defaultPassword = 'admin123'; // Default password
            try {
                $adminModel = new Admin();
                
                // Generate default email - use hospital code or hospital ID as fallback
                $hospitalCode = trim($data['code'] ?? '');
                if (empty($hospitalCode)) {
                    // If no code, use hospital ID
                    $hospitalCode = 'hospital' . $id;
                } else {
                    // Clean code: remove special characters, keep only alphanumeric
                    $hospitalCode = strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $hospitalCode));
                    if (empty($hospitalCode)) {
                        $hospitalCode = 'hospital' . $id;
                    }
                }
                
                $defaultEmail = $hospitalCode . '@hospital.local';
                $defaultName = trim($data['name']) . ' Administrator';
                
                // Check if admin with this email already exists
                $existingAdmin = $adminModel->findByEmail($defaultEmail);
                if (!$existingAdmin) {
                    $adminData = [
                        'name' => $defaultName,
                        'email' => $defaultEmail,
                        'role' => 'admin',
                        'hospital_id' => $id,
                        'password_hash' => password_hash($defaultPassword, PASSWORD_BCRYPT),
                    ];
                    $adminId = $adminModel->create($adminData);
                    error_log("Auto-created admin user for hospital ID {$id}: {$defaultEmail} (Admin ID: {$adminId})");
                } else {
                    // Admin exists, update hospital_id if it's different
                    if ($existingAdmin['hospital_id'] != $id) {
                        $updateStmt = $adminModel->db()->prepare("UPDATE admins SET hospital_id = :hospital_id WHERE id = :id");
                        $updateStmt->execute([':hospital_id' => $id, ':id' => $existingAdmin['id']]);
                    }
                    error_log("Admin user already exists for hospital ID {$id}: {$defaultEmail}");
                }
            } catch (Throwable $adminError) {
                // Log error but don't fail hospital creation
                error_log('Error auto-creating admin user: ' . $adminError->getMessage());
                $defaultEmail = 'admin@hospital.local'; // Fallback for message
            }

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $successMsg = 'Hospital added successfully!';
            if ($defaultEmail) {
                $successMsg .= ' Admin Login - Email: ' . htmlspecialchars($defaultEmail) . ' | Password: ' . htmlspecialchars($defaultPassword);
            }
            $_SESSION['flash_success'] = $successMsg;
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospitals/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            error_log('Hospital creation error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not add hospital: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/hospitals/add');
            exit;
        }
    }

    public function viewHospital(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid hospital id';
            return;
        }
        
        $model = new Hospital();
        $hospital = $model->find($id);
        if (!$hospital) {
            http_response_code(404);
            echo 'Hospital not found';
            return;
        }
        
        $stats = $model->getStats($id);
        
        // Get hospital admins
        $hospitalAdmins = [];
        try {
            $adminModel = new Admin();
            $stmt = $adminModel->db()->prepare("
                SELECT id, name, email, role, hospital_id, created_at 
                FROM admins 
                WHERE hospital_id = ? AND role != 'super_admin'
                ORDER BY created_at DESC
            ");
            $stmt->execute([$id]);
            $hospitalAdmins = $stmt->fetchAll();
        } catch (Throwable $e) {
            error_log('Error loading hospital admins: ' . $e->getMessage());
            $hospitalAdmins = [];
        }
        
        $this->view('hospitals/view', [
            'title' => 'Hospital Details',
            'base' => defined('BASE_URL') ? BASE_URL : '/hms_project/superadmin',
            'hospital' => $hospital,
            'stats' => $stats,
            'hospitalAdmins' => $hospitalAdmins,
        ]);
    }

    public function editHospital(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid hospital id';
            return;
        }
        
        $model = new Hospital();
        $hospital = $model->find($id);
        if (!$hospital) {
            http_response_code(404);
            echo 'Hospital not found';
            return;
        }
        
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        $formError = $_SESSION['form_error'] ?? null;
        $formErrors = $_SESSION['form_errors'] ?? [];
        $old = $_SESSION['old'] ?? $hospital;
        if (isset($_SESSION['form_error'])) { unset($_SESSION['form_error']); }
        if (isset($_SESSION['form_errors'])) { unset($_SESSION['form_errors']); }
        if (isset($_SESSION['old'])) { unset($_SESSION['old']); }

        $this->view('hospitals/edit', [
            'title' => 'Edit Hospital',
            'hospital' => $hospital,
            'formError' => $formError,
            'formErrors' => $formErrors,
            'old' => $old,
        ]);
    }

    public function updateHospital(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid hospital id';
            return;
        }

        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'code' => trim($_POST['code'] ?? ''),
            'address' => trim($_POST['address'] ?? ''),
            'city' => trim($_POST['city'] ?? ''),
            'state' => trim($_POST['state'] ?? ''),
            'country' => trim($_POST['country'] ?? 'India'),
            'pincode' => trim($_POST['pincode'] ?? ''),
            'phone' => trim($_POST['phone'] ?? ''),
            'email' => trim($_POST['email'] ?? ''),
            'website' => trim($_POST['website'] ?? ''),
            'registration_number' => trim($_POST['registration_number'] ?? ''),
            'license_number' => trim($_POST['license_number'] ?? ''),
            'status' => trim($_POST['status'] ?? 'Active'),
            'subscription_plan' => trim($_POST['subscription_plan'] ?? 'Basic'),
            'max_users' => isset($_POST['max_users']) ? (int)$_POST['max_users'] : 10,
            'max_patients' => isset($_POST['max_patients']) ? (int)$_POST['max_patients'] : null,
        ];

        $errors = $this->validateHospital($data);

        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['old'] = $data;
            $_SESSION['form_errors'] = $errors;
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/hospitals/edit?id=' . urlencode((string)$id));
            exit;
        }

        try {
            $model = new Hospital();
            $model->update($id, $data);

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Hospital updated successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospitals/view?id=' . urlencode((string)$id));
            exit;
        } catch (Throwable $e) {
            error_log('Hospital update error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not update hospital: ' . $e->getMessage();
            $base = (defined('BASE_URL') ? BASE_URL : '');
            header('Location: ' . $base . '/hospitals/edit?id=' . urlencode((string)$id));
            exit;
        }
    }

    public function deleteHospital(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) {
            http_response_code(422);
            echo 'Invalid hospital id';
            return;
        }

        try {
            $model = new Hospital();
            $hospital = $model->find($id);
            
            if (!$hospital) {
                if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
                $_SESSION['form_error'] = 'Hospital not found.';
                header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospitals');
                exit;
            }

            // Check if hospital has associated data
            $stats = $model->getStats($id);
            $hasData = ($stats['patients'] > 0 || $stats['doctors'] > 0 || $stats['appointments'] > 0 || $stats['admins'] > 0);
            
            if ($hasData) {
                // Delete associated data first (cascade delete)
                $pdo = $model->db();
                
                // Delete in order: appointments -> admissions -> patients/doctors -> admins
                try {
                    // Delete appointments
                    $pdo->prepare("DELETE FROM appointments WHERE hospital_id = ?")->execute([$id]);
                } catch (Throwable $e) { error_log('Error deleting appointments: ' . $e->getMessage()); }
                
                try {
                    // Delete admissions
                    $pdo->prepare("DELETE FROM admissions WHERE hospital_id = ?")->execute([$id]);
                } catch (Throwable $e) { error_log('Error deleting admissions: ' . $e->getMessage()); }
                
                try {
                    // Delete patients
                    $pdo->prepare("DELETE FROM patients WHERE hospital_id = ?")->execute([$id]);
                } catch (Throwable $e) { error_log('Error deleting patients: ' . $e->getMessage()); }
                
                try {
                    // Delete doctors
                    $pdo->prepare("DELETE FROM doctors WHERE hospital_id = ?")->execute([$id]);
                } catch (Throwable $e) { error_log('Error deleting doctors: ' . $e->getMessage()); }
                
                try {
                    // Delete hospital admins (but not super admin)
                    $pdo->prepare("DELETE FROM admins WHERE hospital_id = ? AND role != 'super_admin'")->execute([$id]);
                } catch (Throwable $e) { error_log('Error deleting admins: ' . $e->getMessage()); }
            }
            
            // Delete the hospital
            $model->delete($id);

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Hospital "' . htmlspecialchars($hospital['name'] ?? '') . '" and all associated data deleted successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospitals');
            exit;
        } catch (Throwable $e) {
            error_log('Hospital deletion error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not delete hospital: ' . $e->getMessage();
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospitals');
            exit;
        }
    }

    public function hospitalAdmins(): void
    {
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        
        $hospitalId = isset($_GET['hospital_id']) ? (int)$_GET['hospital_id'] : null;
        $q = trim((string)($_GET['q'] ?? ''));
        $page = max(1, (int)($_GET['page'] ?? 1));
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $adminModel = new Admin();
        $hospitalModel = new Hospital();
        
        $where = ["role != 'super_admin'"];
        $params = [];
        
        if ($hospitalId !== null) {
            $where[] = 'hospital_id = :hospital_id';
            $params[':hospital_id'] = $hospitalId;
        }
        
        if ($q !== '') {
            $where[] = '(name LIKE :q OR email LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        
        $sql = "SELECT a.id, a.name, a.email, a.role, a.hospital_id, a.created_at, h.name AS hospital_name 
                FROM admins a 
                LEFT JOIN hospitals h ON h.id = a.hospital_id 
                WHERE " . implode(' AND ', $where) . "
                ORDER BY a.created_at DESC 
                LIMIT :limit OFFSET :offset";
        
        $stmt = $adminModel->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $admins = $stmt->fetchAll();
        
        // Count total
        $countSql = "SELECT COUNT(*) FROM admins WHERE " . implode(' AND ', $where);
        $countStmt = $adminModel->db()->prepare($countSql);
        foreach ($params as $key => $value) {
            $countStmt->bindValue($key, $value);
        }
        $countStmt->execute();
        $total = (int)$countStmt->fetchColumn();

        $flashSuccess = $_SESSION['flash_success'] ?? null;
        if (isset($_SESSION['flash_success'])) { unset($_SESSION['flash_success']); }
        
        $this->view('hospital-admins/list', [
            'title' => 'Hospital Admins',
            'base' => defined('BASE_URL') ? BASE_URL : '/hms_project/superadmin',
            'admins' => $admins,
            'hospitals' => $hospitalModel->all(),
            'selectedHospital' => $hospitalId,
            'q' => $q,
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total,
            'flashSuccess' => $flashSuccess,
        ]);
    }

    public function addHospitalAdmin(): void
    {
        $hospitalModel = new Hospital();
        
        $this->view('hospital-admins/add', [
            'title' => 'Add Hospital Admin',
            'hospitals' => $hospitalModel->all(),
        ]);
    }

    public function storeHospitalAdmin(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $name = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $hospitalId = isset($_POST['hospital_id']) ? (int)$_POST['hospital_id'] : 0;

        $errors = [];
        if (empty($name)) $errors['name'] = 'Name is required.';
        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors['email'] = 'Valid email is required.';
        if (empty($password) || strlen($password) < 6) $errors['password'] = 'Password must be at least 6 characters.';
        if ($hospitalId <= 0) $errors['hospital_id'] = 'Hospital is required.';

        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_errors'] = $errors;
            $_SESSION['old'] = $_POST;
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospital-admins/add');
            exit;
        }

        try {
            $adminModel = new Admin();
            
            // Check if email exists
            $existing = $adminModel->findByEmail($email);
            if ($existing) {
                throw new Exception('Email already exists.');
            }
            
            $passwordHash = password_hash($password, PASSWORD_DEFAULT);
            $stmt = $adminModel->db()->prepare("
                INSERT INTO admins (name, email, role, hospital_id, password_hash, created_at) 
                VALUES (:name, :email, 'admin', :hospital_id, :password_hash, NOW())
            ");
            $stmt->execute([
                ':name' => $name,
                ':email' => $email,
                ':hospital_id' => $hospitalId,
                ':password_hash' => $passwordHash,
            ]);

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Hospital admin created successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospital-admins');
            exit;
        } catch (Throwable $e) {
            error_log('Admin creation error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not create admin: ' . $e->getMessage();
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospital-admins/add');
            exit;
        }
    }

    public function editHospitalAdmin(): void
    {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($id <= 0) {
            http_response_code(400);
            echo 'Invalid admin id';
            return;
        }
        
        $adminModel = new Admin();
        $stmt = $adminModel->db()->prepare("
            SELECT a.*, h.name AS hospital_name 
            FROM admins a 
            LEFT JOIN hospitals h ON h.id = a.hospital_id 
            WHERE a.id = :id
        ");
        $stmt->execute([':id' => $id]);
        $admin = $stmt->fetch();
        
        if (!$admin) {
            http_response_code(404);
            echo 'Admin not found';
            return;
        }
        
        if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
        
        $formError = $_SESSION['form_error'] ?? null;
        $formErrors = $_SESSION['form_errors'] ?? [];
        $old = $_SESSION['old'] ?? $admin;
        if (isset($_SESSION['form_error'])) { unset($_SESSION['form_error']); }
        if (isset($_SESSION['form_errors'])) { unset($_SESSION['form_errors']); }
        if (isset($_SESSION['old'])) { unset($_SESSION['old']); }
        
        $hospitalModel = new Hospital();
        
        $this->view('hospital-admins/edit', [
            'title' => 'Edit Hospital Admin',
            'base' => defined('BASE_URL') ? BASE_URL : '/hms_project/superadmin',
            'admin' => $admin,
            'hospitals' => $hospitalModel->all(),
            'formError' => $formError,
            'formErrors' => $formErrors,
            'old' => $old,
        ]);
    }

    public function updateHospitalAdmin(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        $name = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $hospitalId = isset($_POST['hospital_id']) ? (int)$_POST['hospital_id'] : 0;

        $errors = [];
        if (empty($name)) $errors['name'] = 'Name is required.';
        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors['email'] = 'Valid email is required.';
        if ($hospitalId <= 0) $errors['hospital_id'] = 'Hospital is required.';

        if (!empty($errors)) {
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_errors'] = $errors;
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospital-admins/edit?id=' . $id);
            exit;
        }

        try {
            $adminModel = new Admin();
            
            // Check if email exists for another user
            $existing = $adminModel->findByEmail($email);
            if ($existing && (int)$existing['id'] !== $id) {
                throw new Exception('Email already exists.');
            }
            
            $sql = "UPDATE admins SET name = :name, email = :email, hospital_id = :hospital_id";
            $params = [
                ':id' => $id,
                ':name' => $name,
                ':email' => $email,
                ':hospital_id' => $hospitalId,
            ];
            
            if (!empty($password)) {
                if (strlen($password) < 6) {
                    throw new Exception('Password must be at least 6 characters.');
                }
                $sql .= ", password_hash = :password_hash";
                $params[':password_hash'] = password_hash($password, PASSWORD_DEFAULT);
            }
            
            $sql .= " WHERE id = :id";
            $params[':id'] = $id;
            
            $stmt = $adminModel->db()->prepare($sql);
            $stmt->execute($params);

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Hospital admin updated successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospital-admins');
            exit;
        } catch (Throwable $e) {
            error_log('Admin update error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not update admin: ' . $e->getMessage();
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospital-admins/edit?id=' . $id);
            exit;
        }
    }

    public function deleteHospitalAdmin(): void
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            http_response_code(405);
            echo 'Method Not Allowed';
            return;
        }

        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) {
            http_response_code(422);
            echo 'Invalid admin id';
            return;
        }

        try {
            $adminModel = new Admin();
            $stmt = $adminModel->db()->prepare("DELETE FROM admins WHERE id = :id AND role != 'super_admin'");
            $stmt->execute([':id' => $id]);

            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['flash_success'] = 'Hospital admin deleted successfully.';
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospital-admins');
            exit;
        } catch (Throwable $e) {
            error_log('Admin deletion error: ' . $e->getMessage());
            if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
            $_SESSION['form_error'] = 'Could not delete admin: ' . $e->getMessage();
            header('Location: ' . (defined('BASE_URL') ? BASE_URL : '') . '/hospital-admins');
            exit;
        }
    }

    protected function validateHospital(array $data): array
    {
        $errors = [];

        if (empty($data['name'])) {
            $errors['name'] = 'Hospital name is required.';
        }
        if (empty($data['code'])) {
            $errors['code'] = 'Hospital code is required.';
        }
        if (!empty($data['email']) && !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Invalid email format.';
        }

        return $errors;
    }
}

