<?php
declare(strict_types=1);

class Hospital extends Model
{
    public function all(): array
    {
        $sql = "SELECT * FROM hospitals ORDER BY created_at DESC";
        $stmt = $this->db()->query($sql);
        return $stmt->fetchAll();
    }

    public function list(array $filters = []): array
    {
        $q = $filters['q'] ?? '';
        $status = $filters['status'] ?? null;
        $limit = $filters['limit'] ?? 20;
        $offset = $filters['offset'] ?? 0;

        $where = [];
        $params = [];

        $sql = "SELECT * FROM hospitals WHERE 1=1";

        if ($q !== '') {
            $where[] = '(name LIKE :q OR code LIKE :q OR city LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($status !== null) {
            $where[] = 'status = :status';
            $params[':status'] = $status;
        }

        if (!empty($where)) {
            $sql .= ' AND ' . implode(' AND ', $where);
        }

        $sql .= " ORDER BY created_at DESC LIMIT :limit OFFSET :offset";

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function count(array $filters = []): int
    {
        $q = $filters['q'] ?? '';
        $status = $filters['status'] ?? null;

        $where = [];
        $params = [];

        $sql = "SELECT COUNT(*) FROM hospitals WHERE 1=1";

        if ($q !== '') {
            $where[] = '(name LIKE :q OR code LIKE :q OR city LIKE :q)';
            $params[':q'] = '%' . $q . '%';
        }
        if ($status !== null) {
            $where[] = 'status = :status';
            $params[':status'] = $status;
        }

        if (!empty($where)) {
            $sql .= ' AND ' . implode(' AND ', $where);
        }

        $stmt = $this->db()->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function find(int $id): ?array
    {
        $sql = "SELECT * FROM hospitals WHERE id = :id";
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([':id' => $id]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public function create(array $data): int
    {
        $sql = "INSERT INTO hospitals (
            name, code, address, city, state, country, pincode, 
            phone, email, website, registration_number, license_number, 
            logo, status, subscription_plan, subscription_expires_at, 
            max_users, max_patients, created_at
        ) VALUES (
            :name, :code, :address, :city, :state, :country, :pincode,
            :phone, :email, :website, :registration_number, :license_number,
            :logo, :status, :subscription_plan, :subscription_expires_at,
            :max_users, :max_patients, NOW()
        )";
        
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':name' => $data['name'] ?? null,
            ':code' => $data['code'] ?? null,
            ':address' => $data['address'] ?? null,
            ':city' => $data['city'] ?? null,
            ':state' => $data['state'] ?? null,
            ':country' => $data['country'] ?? 'India',
            ':pincode' => $data['pincode'] ?? null,
            ':phone' => $data['phone'] ?? null,
            ':email' => $data['email'] ?? null,
            ':website' => $data['website'] ?? null,
            ':registration_number' => $data['registration_number'] ?? null,
            ':license_number' => $data['license_number'] ?? null,
            ':logo' => $data['logo'] ?? null,
            ':status' => $data['status'] ?? 'Active',
            ':subscription_plan' => $data['subscription_plan'] ?? 'Basic',
            ':subscription_expires_at' => $data['subscription_expires_at'] ?? null,
            ':max_users' => isset($data['max_users']) ? (int)$data['max_users'] : 10,
            ':max_patients' => isset($data['max_patients']) ? (int)$data['max_patients'] : null,
        ]);
        return (int)$this->db()->lastInsertId();
    }

    public function update(int $id, array $data): void
    {
        $sql = "UPDATE hospitals SET 
            name = :name, code = :code, address = :address, city = :city, 
            state = :state, country = :country, pincode = :pincode,
            phone = :phone, email = :email, website = :website, 
            registration_number = :registration_number, license_number = :license_number,
            logo = :logo, status = :status, subscription_plan = :subscription_plan, 
            subscription_expires_at = :subscription_expires_at,
            max_users = :max_users, max_patients = :max_patients,
            updated_at = NOW()
        WHERE id = :id";
        
        $stmt = $this->db()->prepare($sql);
        $stmt->execute([
            ':id' => $id,
            ':name' => $data['name'] ?? null,
            ':code' => $data['code'] ?? null,
            ':address' => $data['address'] ?? null,
            ':city' => $data['city'] ?? null,
            ':state' => $data['state'] ?? null,
            ':country' => $data['country'] ?? 'India',
            ':pincode' => $data['pincode'] ?? null,
            ':phone' => $data['phone'] ?? null,
            ':email' => $data['email'] ?? null,
            ':website' => $data['website'] ?? null,
            ':registration_number' => $data['registration_number'] ?? null,
            ':license_number' => $data['license_number'] ?? null,
            ':logo' => $data['logo'] ?? null,
            ':status' => $data['status'] ?? 'Active',
            ':subscription_plan' => $data['subscription_plan'] ?? 'Basic',
            ':subscription_expires_at' => $data['subscription_expires_at'] ?? null,
            ':max_users' => isset($data['max_users']) ? (int)$data['max_users'] : 10,
            ':max_patients' => isset($data['max_patients']) ? (int)$data['max_patients'] : null,
        ]);
    }

    public function delete(int $id): void
    {
        $stmt = $this->db()->prepare("DELETE FROM hospitals WHERE id = :id");
        $stmt->execute([':id' => $id]);
    }

    public function getStats(int $hospitalId): array
    {
        $stats = [
            'patients' => 0,
            'doctors' => 0,
            'appointments' => 0,
            'admins' => 0,
        ];
        
        try {
            // Count patients for this hospital
            $stmt = $this->db()->prepare("SELECT COUNT(*) FROM patients WHERE hospital_id = :id");
            $stmt->execute([':id' => $hospitalId]);
            $stats['patients'] = (int)$stmt->fetchColumn();
        } catch (Throwable $e) {
            error_log("Error counting patients for hospital {$hospitalId}: " . $e->getMessage());
            // Try without hospital_id filter if column doesn't exist
            try {
                $stmt = $this->db()->query("SELECT COUNT(*) FROM patients");
                $stats['patients'] = (int)$stmt->fetchColumn();
            } catch (Throwable $e2) {
                // Ignore
            }
        }
        
        try {
            // Count doctors for this hospital
            $stmt = $this->db()->prepare("SELECT COUNT(*) FROM doctors WHERE hospital_id = :id");
            $stmt->execute([':id' => $hospitalId]);
            $stats['doctors'] = (int)$stmt->fetchColumn();
        } catch (Throwable $e) {
            error_log("Error counting doctors for hospital {$hospitalId}: " . $e->getMessage());
            // Try without hospital_id filter if column doesn't exist
            try {
                $stmt = $this->db()->query("SELECT COUNT(*) FROM doctors");
                $stats['doctors'] = (int)$stmt->fetchColumn();
            } catch (Throwable $e2) {
                // Ignore
            }
        }
        
        try {
            // Count appointments for this hospital
            $stmt = $this->db()->prepare("SELECT COUNT(*) FROM appointments WHERE hospital_id = :id");
            $stmt->execute([':id' => $hospitalId]);
            $stats['appointments'] = (int)$stmt->fetchColumn();
        } catch (Throwable $e) {
            error_log("Error counting appointments for hospital {$hospitalId}: " . $e->getMessage());
            // Try without hospital_id filter if column doesn't exist
            try {
                $stmt = $this->db()->query("SELECT COUNT(*) FROM appointments");
                $stats['appointments'] = (int)$stmt->fetchColumn();
            } catch (Throwable $e2) {
                // Ignore
            }
        }
        
        try {
            // Count admins for this hospital
            $stmt = $this->db()->prepare("SELECT COUNT(*) FROM admins WHERE hospital_id = :id AND role != 'super_admin'");
            $stmt->execute([':id' => $hospitalId]);
            $stats['admins'] = (int)$stmt->fetchColumn();
        } catch (Throwable $e) {
            error_log("Error counting admins for hospital {$hospitalId}: " . $e->getMessage());
        }
        
        return $stats;
    }
}

