<?php
declare(strict_types=1);

class App
{
    protected array $routes = [];

    public function get(string $path, string $handler): void
    {
        $this->addRoute('GET', $path, $handler);
    }

    public function post(string $path, string $handler): void
    {
        $this->addRoute('POST', $path, $handler);
    }

    public function addRoute(string $method, string $path, string $handler): void
    {
        $this->routes[] = [
            'method' => strtoupper($method),
            'path'   => '/' . trim($path, '/'),
            'handler'=> $handler,
        ];
    }

    public function dispatch(string $basePath = ''): void
    {
        $requestMethod = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        $requestUri = $_SERVER['REQUEST_URI'] ?? '/';

        // strip query string
        if (strpos($requestUri, '?') !== false) {
            $requestUri = strstr($requestUri, '?', true);
        }

        // normalize path and remove basePath if provided
        $path = '/' . trim($requestUri, '/');
        $basePath = '/' . trim($basePath, '/');
        if ($basePath !== '/' && str_starts_with($path, $basePath)) {
            $path = substr($path, strlen($basePath));
            $path = '/' . ltrim($path, '/');
        }
        if ($path === '//') { $path = '/'; }

        foreach ($this->routes as $route) {
            if ($route['method'] !== $requestMethod) { continue; }
            if ($route['path'] === $path) {
                $this->invokeHandler($route['handler']);
                return;
            }
        }

        http_response_code(404);
        echo '404 Not Found';
    }

    protected function invokeHandler(string $handler): void
    {
        // Handler format: Controller@method
        if (!str_contains($handler, '@')) {
            throw new RuntimeException('Invalid handler: ' . $handler);
        }
        [$controllerName, $method] = explode('@', $handler, 2);

        if (!class_exists($controllerName)) {
            throw new RuntimeException("Controller not found: {$controllerName}");
        }
        $controller = new $controllerName();
        if (!method_exists($controller, $method)) {
            throw new RuntimeException("Method not found: {$controllerName}::{$method}");
        }
        $controller->$method();
    }
}

