<?php
declare(strict_types=1);

class Controller
{
    protected function view(string $viewPath, array $data = []): void
    {
        $view = new View();
        // Pass hospitalName and role to all views
        $data['hospitalName'] = $this->getHospitalName();
        $data['role'] = Auth::user()['role'] ?? 'Guest';
        $view->render($viewPath, $data);
    }

    /**
     * Get current hospital ID from session (for multi-hospital support)
     * Returns null if user is super admin or hospital_id not set
     */
    protected function getHospitalId(): ?int
    {
        if (!class_exists('Auth')) {
            return null;
        }
        
        $user = Auth::user();
        if (!$user) {
            return null;
        }

        // Super admin has no hospital_id
        if (($user['role'] ?? '') === 'super_admin') {
            return null;
        }

        return isset($user['hospital_id']) ? (int)$user['hospital_id'] : null;
    }

    /**
     * Check if current user is super admin
     */
    protected function isSuperAdmin(): bool
    {
        if (!class_exists('Auth')) {
            return false;
        }
        
        $user = Auth::user();
        return ($user['role'] ?? '') === 'super_admin';
    }

    /**
     * Check if current user is hospital admin
     */
    protected function isHospitalAdmin(): bool
    {
        if (!class_exists('Auth')) {
            return false;
        }
        
        $user = Auth::user();
        $role = $user['role'] ?? '';
        return $role === 'admin' && !empty($user['hospital_id']);
    }

    /**
     * Get hospital name for current user
     * Returns hospital name if user is hospital admin, otherwise returns default
     */
    protected function getHospitalName(): string
    {
        if ($this->isSuperAdmin()) {
            return 'Super Admin Panel';
        }
        
        $hospitalId = $this->getHospitalId();
        if (!$hospitalId) {
            return 'Hospital Management System';
        }
        
        try {
            // Determine base path
            $basePath = defined('BASE_PATH') ? BASE_PATH : dirname(__DIR__);
            
            // Try to load Hospital model from admin app
            $hospitalModelPath = $basePath . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'Hospital.php';
            if (file_exists($hospitalModelPath)) {
                require_once $hospitalModelPath;
            }
            
            // Also try superadmin app
            $superAdminHospitalPath = $basePath . DIRECTORY_SEPARATOR . 'superadmin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'Hospital.php';
            if (!class_exists('Hospital') && file_exists($superAdminHospitalPath)) {
                require_once $superAdminHospitalPath;
            }
            
            if (class_exists('Hospital')) {
                $hospitalModel = new Hospital();
                $hospital = $hospitalModel->find($hospitalId);
                return $hospital['name'] ?? 'Hospital Management System';
            }
        } catch (Throwable $e) {
            error_log('Error fetching hospital name: ' . $e->getMessage());
        }
        
        return 'Hospital Management System';
    }
}

