<?php
declare(strict_types=1);

class View
{
    public function render(string $viewPath, array $data = []): void
    {
        if (!defined('APP_PATH')) {
            throw new RuntimeException('APP_PATH is not defined. Define it in your front controller.');
        }

        $viewsDir = rtrim(APP_PATH, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'views' . DIRECTORY_SEPARATOR;
        $normalized = str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $viewPath);
        
        // Check in html subfolder first (module-wise structure)
        $parts = explode(DIRECTORY_SEPARATOR, $normalized);
        if (count($parts) >= 2) {
            $moduleName = $parts[0];
            $viewName = implode(DIRECTORY_SEPARATOR, array_slice($parts, 1));
            $htmlSubfolderPath = $viewsDir . $moduleName . DIRECTORY_SEPARATOR . 'html' . DIRECTORY_SEPARATOR . $viewName;
            $phpFileHtml = $htmlSubfolderPath . '.php';
            $htmlFileHtml = $htmlSubfolderPath . '.html';
            if (is_file($phpFileHtml)) {
                $phpFile = $phpFileHtml;
                $htmlFile = $htmlFileHtml;
            } else {
                $phpFile = $viewsDir . $normalized . '.php';
                $htmlFile = $viewsDir . $normalized . '.html';
            }
        } else {
            $phpFile = $viewsDir . $normalized . '.php';
            $htmlFile = $viewsDir . $normalized . '.html';
        }

        // Provide base URL to all views
        if (!array_key_exists('base', $data)) {
            $data['base'] = defined('BASE_URL') ? (string)BASE_URL : '';
        }

        if (is_file($htmlFile)) {
            echo $this->renderHtmlTemplate($htmlFile, $viewsDir, $data);
            return;
        }

        if (is_file($phpFile)) {
            extract($data, EXTR_SKIP);
            require $phpFile;
            return;
        }

        throw new RuntimeException('View not found: ' . $htmlFile . ' or ' . $phpFile);
    }

    protected function renderHtmlTemplate(string $file, string $viewsDir, array $data): string
    {
        $template = (string)file_get_contents($file);
        if ($template === '') { return ''; }

        // Partials: {{> include/navbar}} resolves to viewsDir/include/navbar.html (or .php fallback rendered to string)
        $template = preg_replace_callback('/\{\{>\s*([a-zA-Z0-9_\-\/\.]+)\s*\}\}/', function ($m) use ($viewsDir, $data) {
            $partialPath = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $m[1]);
            $partialHtml = $viewsDir . $partialPath . '.html';
            $partialPhp  = $viewsDir . $partialPath . '.php';
            if (is_file($partialHtml)) {
                return $this->renderHtmlTemplate($partialHtml, $viewsDir, $data);
            }
            if (is_file($partialPhp)) {
                // Capture PHP include output
                ob_start();
                extract($data, EXTR_SKIP);
                require $partialPhp;
                return (string)ob_get_clean();
            }
            return '';
        }, $template);

        // Triple braces {{{var}}} for raw output (no escaping)
        $template = preg_replace_callback('/\{\{\{\s*([a-zA-Z0-9_\.]+)\s*\}\}\}/', function ($m) use ($data) {
            $val = $this->resolve($data, $m[1]);
            return is_scalar($val) ? (string)$val : '';
        }, $template);

        // Each blocks: {{#each items}} ... {{/each}}
        $template = preg_replace_callback('/\{\{#each\s+([a-zA-Z0-9_\.]+)\s*\}\}([\s\S]*?)\{\{\/each\}\}/', function ($m) use ($viewsDir, $data) {
            $list = $this->resolve($data, $m[1]);
            if (!is_iterable($list)) { return ''; }
            $inner = $m[2];
            $out = '';
            foreach ($list as $item) {
                // For each iteration, merge item over parent for lookups
                $ctx = is_array($item) ? array_merge($data, $item) : array_merge($data, ['this' => $item]);
                $out .= $this->renderInline($inner, $viewsDir, $ctx);
            }
            return $out;
        }, $template);

        // Double braces {{var}} for escaped output
        $template = preg_replace_callback('/\{\{\s*([a-zA-Z0-9_\.]+)\s*\}\}/', function ($m) use ($data) {
            $val = $this->resolve($data, $m[1]);
            return htmlspecialchars(is_scalar($val) ? (string)$val : '', ENT_QUOTES, 'UTF-8');
        }, $template);

        return $template;
    }

    protected function renderInline(string $tpl, string $viewsDir, array $data): string
    {
        // Support nested partials inside loops
        $tpl = preg_replace_callback('/\{\{>\s*([a-zA-Z0-9_\-\/\.]+)\s*\}\}/', function ($m) use ($viewsDir, $data) {
            $partialPath = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $m[1]);
            $partialHtml = $viewsDir . $partialPath . '.html';
            $partialPhp  = $viewsDir . $partialPath . '.php';
            if (is_file($partialHtml)) {
                return $this->renderHtmlTemplate($partialHtml, $viewsDir, $data);
            }
            if (is_file($partialPhp)) {
                ob_start();
                extract($data, EXTR_SKIP);
                require $partialPhp;
                return (string)ob_get_clean();
            }
            return '';
        }, $tpl);

        // Raw
        $tpl = preg_replace_callback('/\{\{\{\s*([a-zA-Z0-9_\.]+)\s*\}\}\}/', function ($m) use ($data) {
            $val = $this->resolve($data, $m[1]);
            return is_scalar($val) ? (string)$val : '';
        }, $tpl);
        // Escaped
        $tpl = preg_replace_callback('/\{\{\s*([a-zA-Z0-9_\.]+)\s*\}\}/', function ($m) use ($data) {
            $val = $this->resolve($data, $m[1]);
            return htmlspecialchars(is_scalar($val) ? (string)$val : '', ENT_QUOTES, 'UTF-8');
        }, $tpl);
        return $tpl;
    }

    protected function resolve(array $data, string $path)
    {
        if ($path === '') { return null; }
        $parts = explode('.', $path);
        $cur = $data;
        foreach ($parts as $p) {
            if (is_array($cur) && array_key_exists($p, $cur)) {
                $cur = $cur[$p];
            } else {
                return null;
            }
        }
        return $cur;
    }
}

