<?php
/**
 * Update Existing Data with Hospital ID = 1
 * This script will set hospital_id = 1 for all existing data
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<!DOCTYPE html><html><head><title>Update Hospital ID</title>";
echo "<style>body{font-family:Arial;padding:20px;background:#f5f5f5;}pre{background:#fff;padding:15px;border-radius:5px;box-shadow:0 2px 5px rgba(0,0,0,0.1);}</style>";
echo "</head><body>";
echo "<h2>Update Existing Data with Hospital ID = 1</h2>";
echo "<pre>";

// Load database config
$dbConfigFile = __DIR__ . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'database.php';
if (!file_exists($dbConfigFile)) {
    die("✗ Database config file not found\n");
}

$dbConfig = require $dbConfigFile;

// Extract connection details
$host = '127.0.0.1';
$port = 3306;
$dbname = 'hms';
$user = $dbConfig['user'] ?? 'root';
$pass = $dbConfig['pass'] ?? '';

if (isset($dbConfig['dsn'])) {
    if (preg_match('/host=([^;]+)/', $dbConfig['dsn'], $matches)) {
        $host = $matches[1];
    }
    if (preg_match('/port=(\d+)/', $dbConfig['dsn'], $matches)) {
        $port = (int)$matches[1];
    }
    if (preg_match('/dbname=([^;]+)/', $dbConfig['dsn'], $matches)) {
        $dbname = $matches[1];
    }
}

try {
    // Connect to database
    echo "Connecting to database...\n";
    $pdo = new PDO(
        "mysql:host={$host};port={$port};dbname={$dbname};charset=utf8mb4",
        $user,
        $pass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    echo "✓ Connected to database '{$dbname}'\n\n";

    // Check if hospitals table exists and get first hospital
    echo "Step 1: Checking hospitals...\n";
    $stmt = $pdo->query("SELECT id, name FROM hospitals ORDER BY id ASC LIMIT 1");
    $hospital = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$hospital) {
        echo "⚠ No hospital found. Creating default hospital...\n";
        $stmt = $pdo->prepare("
            INSERT INTO hospitals (name, code, status, created_at) 
            VALUES (?, 'HOSP-001', 'Active', NOW())
        ");
        $stmt->execute(['Default Hospital']);
        $hospitalId = (int)$pdo->lastInsertId();
        echo "✓ Created default hospital with ID: {$hospitalId}\n\n";
    } else {
        $hospitalId = (int)$hospital['id'];
        echo "✓ Found hospital: {$hospital['name']} (ID: {$hospitalId})\n\n";
    }

    // Tables to update
    $tables = [
        'patients' => 'Patient records',
        'doctors' => 'Doctor records',
        'departments' => 'Department records',
        'appointments' => 'Appointment records',
        'rooms' => 'Room records',
        'beds' => 'Bed records',
        'admissions' => 'Admission records',
        'invoices' => 'Invoice records',
        'medicines' => 'Medicine records',
        'pharmacy_sales' => 'Pharmacy sales',
        'lab_tests' => 'Lab test records',
        'lab_orders' => 'Lab order records',
        'services' => 'Service records',
        'charges' => 'Charge records',
        'hr_staff' => 'HR staff records',
    ];

    $updated = [];
    $skipped = [];

    echo "Step 2: Updating existing data...\n\n";

    foreach ($tables as $table => $description) {
        // Check if table exists
        $stmt = $pdo->query("SHOW TABLES LIKE '{$table}'");
        if (!$stmt->fetch()) {
            echo "  ⊙ {$table} - Table does not exist, skipping...\n";
            $skipped[] = $table;
            continue;
        }

        // Check if hospital_id column exists
        $stmt = $pdo->query("SHOW COLUMNS FROM {$table} LIKE 'hospital_id'");
        if (!$stmt->fetch()) {
            echo "  ⊙ {$table} - hospital_id column does not exist, skipping...\n";
            $skipped[] = $table;
            continue;
        }

        // Count records with NULL hospital_id
        $stmt = $pdo->query("SELECT COUNT(*) FROM {$table} WHERE hospital_id IS NULL");
        $nullCount = (int)$stmt->fetchColumn();

        if ($nullCount > 0) {
            try {
                $stmt = $pdo->prepare("UPDATE {$table} SET hospital_id = ? WHERE hospital_id IS NULL");
                $stmt->execute([$hospitalId]);
                $affected = $stmt->rowCount();
                echo "  ✓ {$table} - Updated {$affected} record(s)\n";
                $updated[$table] = $affected;
            } catch (PDOException $e) {
                echo "  ✗ {$table} - Error: " . $e->getMessage() . "\n";
                $skipped[] = $table;
            }
        } else {
            echo "  ⊙ {$table} - No records with NULL hospital_id\n";
        }
    }

    // Update admins (set hospital_id for non-super-admin users)
    echo "\nStep 3: Updating admin users...\n";
    $stmt = $pdo->query("SELECT COUNT(*) FROM admins WHERE role != 'super_admin' AND hospital_id IS NULL");
    $nullAdminCount = (int)$stmt->fetchColumn();
    
    if ($nullAdminCount > 0) {
        try {
            $stmt = $pdo->prepare("UPDATE admins SET hospital_id = ? WHERE role != 'super_admin' AND hospital_id IS NULL");
            $stmt->execute([$hospitalId]);
            $affected = $stmt->rowCount();
            echo "  ✓ admins - Updated {$affected} admin user(s) with hospital_id = {$hospitalId}\n";
            $updated['admins'] = $affected;
        } catch (PDOException $e) {
            echo "  ✗ admins - Error: " . $e->getMessage() . "\n";
        }
    } else {
        echo "  ⊙ admins - No admin users to update\n";
    }

    // Summary
    echo "\n";
    echo "========================================\n";
    echo "Update Summary\n";
    echo "========================================\n";
    echo "Hospital ID used: {$hospitalId}\n";
    echo "Tables updated: " . count($updated) . "\n";
    if (!empty($updated)) {
        echo "\nUpdated tables:\n";
        foreach ($updated as $table => $count) {
            echo "  - {$table}: {$count} record(s)\n";
        }
    }
    if (!empty($skipped)) {
        echo "\nSkipped tables: " . count($skipped) . "\n";
    }

    echo "\n";
    echo "========================================\n";
    echo "✓ Update Complete!\n";
    echo "========================================\n";
    echo "\n";
    echo "All existing data has been assigned to Hospital ID: {$hospitalId}\n";
    echo "Super admin can now see this data in the dashboard.\n";
    
} catch (Exception $e) {
    echo "\n";
    echo "========================================\n";
    echo "✗ Error!\n";
    echo "========================================\n";
    echo "Error: " . $e->getMessage() . "\n";
}

echo "</pre>";
echo "</body></html>";
?>

