<?php
/**
 * Script to verify data isolation between hospitals
 * Run via browser: http://localhost/hms_project/verify_data_isolation.php
 */

// Load database configuration
$dbConfigFile = __DIR__ . DIRECTORY_SEPARATOR . 'admin' . DIRECTORY_SEPARATOR . 'app' . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'database.php';
if (!file_exists($dbConfigFile)) {
    die("❌ Database config file not found at: " . htmlspecialchars($dbConfigFile));
}

$dbConfig = require $dbConfigFile;

// Verify config loaded correctly
if (!is_array($dbConfig)) {
    die("❌ Database config file did not return an array");
}

// Extract connection details
$host = '127.0.0.1';
$dbname = 'hms';
$user = isset($dbConfig['user']) ? $dbConfig['user'] : 'root';
$pass = isset($dbConfig['pass']) ? $dbConfig['pass'] : '';

echo "<!DOCTYPE html><html><head><title>Data Isolation Verification</title>";
echo "<style>
    body{font-family:Arial;padding:20px;background:#f5f5f5;}
    .container{max-width:1200px;margin:0 auto;background:#fff;padding:20px;border-radius:8px;box-shadow:0 2px 10px rgba(0,0,0,0.1);}
    h1{color:#333;border-bottom:2px solid #007bff;padding-bottom:10px;}
    h2{color:#555;margin-top:30px;}
    table{border-collapse:collapse;width:100%;margin:20px 0;}
    th,td{padding:12px;text-align:left;border:1px solid #ddd;}
    th{background:#007bff;color:#fff;}
    tr:nth-child(even){background:#f9f9f9;}
    .success{color:#28a745;font-weight:bold;}
    .warning{color:#ffc107;font-weight:bold;}
    .error{color:#dc3545;font-weight:bold;}
    .info{color:#17a2b8;}
    .section{margin:20px 0;padding:15px;background:#f8f9fa;border-radius:5px;}
</style></head><body>";
echo "<div class='container'>";
echo "<h1>🔒 Multi-Hospital Data Isolation Verification</h1>";

try {
    $pdo = new PDO(
        "mysql:host={$host};dbname={$dbname};charset=utf8mb4",
        $user,
        $pass,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );

    echo "<div class='section'>";
    echo "<h2>1. Hospitals Check</h2>";
    
    // Check hospitals
    $stmt = $pdo->query("SELECT id, name, code, status FROM hospitals ORDER BY id");
    $hospitals = $stmt->fetchAll();
    
    if (empty($hospitals)) {
        echo "<p class='warning'>⚠ No hospitals found. Please add hospitals first.</p>";
    } else {
        echo "<table><tr><th>ID</th><th>Name</th><th>Code</th><th>Status</th></tr>";
        foreach ($hospitals as $h) {
            echo "<tr><td>{$h['id']}</td><td>{$h['name']}</td><td>{$h['code']}</td><td>{$h['status']}</td></tr>";
        }
        echo "</table>";
    }
    echo "</div>";

    echo "<div class='section'>";
    echo "<h2>2. Hospital Admin Users</h2>";
    
    $stmt = $pdo->query("
        SELECT a.id, a.name, a.email, a.role, a.hospital_id, h.name AS hospital_name 
        FROM admins a 
        LEFT JOIN hospitals h ON h.id = a.hospital_id 
        WHERE a.role != 'super_admin' 
        ORDER BY a.hospital_id, a.id
    ");
    $admins = $stmt->fetchAll();
    
    if (empty($admins)) {
        echo "<p class='warning'>⚠ No hospital admin users found.</p>";
    } else {
        echo "<table><tr><th>Admin ID</th><th>Name</th><th>Email</th><th>Hospital ID</th><th>Hospital Name</th></tr>";
        foreach ($admins as $a) {
            $hospitalName = $a['hospital_name'] ?? 'N/A';
            $hospitalId = $a['hospital_id'] ?? 'NULL';
            echo "<tr><td>{$a['id']}</td><td>{$a['name']}</td><td>{$a['email']}</td><td>{$hospitalId}</td><td>{$hospitalName}</td></tr>";
        }
        echo "</table>";
    }
    echo "</div>";

    echo "<div class='section'>";
    echo "<h2>3. Data Distribution by Hospital</h2>";
    
    $tables = ['patients', 'doctors', 'appointments', 'admissions', 'departments', 'rooms', 'beds'];
    
    echo "<table><tr><th>Table</th>";
    foreach ($hospitals as $h) {
        echo "<th>Hospital {$h['id']}<br/>({$h['name']})</th>";
    }
    echo "<th>NULL/Total</th></tr>";
    
    foreach ($tables as $table) {
        // Check if table exists and has hospital_id column
        try {
            $stmt = $pdo->query("SHOW COLUMNS FROM `{$table}` LIKE 'hospital_id'");
            $hasColumn = $stmt->fetch();
            
            if ($hasColumn) {
                echo "<tr><td><strong>{$table}</strong></td>";
                
                foreach ($hospitals as $h) {
                    $stmt = $pdo->prepare("SELECT COUNT(*) FROM `{$table}` WHERE hospital_id = ?");
                    $stmt->execute([$h['id']]);
                    $count = $stmt->fetchColumn();
                    $color = $count > 0 ? 'success' : 'info';
                    echo "<td class='{$color}'>{$count}</td>";
                }
                
                // Count NULL or records without hospital_id
                $stmt = $pdo->query("SELECT COUNT(*) FROM `{$table}` WHERE hospital_id IS NULL");
                $nullCount = $stmt->fetchColumn();
                
                $stmt = $pdo->query("SELECT COUNT(*) FROM `{$table}`");
                $totalCount = $stmt->fetchColumn();
                
                $nullColor = $nullCount > 0 ? 'warning' : 'success';
                echo "<td class='{$nullColor}'>{$nullCount} / {$totalCount}</td>";
                echo "</tr>";
            } else {
                echo "<tr><td><strong>{$table}</strong></td>";
                echo "<td colspan='" . (count($hospitals) + 1) . "' class='error'>❌ hospital_id column missing</td></tr>";
            }
        } catch (PDOException $e) {
            echo "<tr><td><strong>{$table}</strong></td>";
            echo "<td colspan='" . (count($hospitals) + 1) . "' class='error'>❌ Table not found or error: " . htmlspecialchars($e->getMessage()) . "</td></tr>";
        }
    }
    
    echo "</table>";
    echo "</div>";

    echo "<div class='section'>";
    echo "<h2>4. Data Isolation Status</h2>";
    
    $allGood = true;
    $issues = [];
    
    // Check each table for hospital_id column
    foreach ($tables as $table) {
        try {
            $stmt = $pdo->query("SHOW COLUMNS FROM `{$table}` LIKE 'hospital_id'");
            if (!$stmt->fetch()) {
                $allGood = false;
                $issues[] = "Table '{$table}' is missing hospital_id column";
            }
        } catch (PDOException $e) {
            // Table might not exist, skip
        }
    }
    
    if ($allGood && empty($issues)) {
        echo "<p class='success'>✅ All tables have hospital_id column for data isolation</p>";
    } else {
        echo "<p class='error'>❌ Issues found:</p><ul>";
        foreach ($issues as $issue) {
            echo "<li>{$issue}</li>";
        }
        echo "</ul>";
    }
    
    echo "</div>";

    echo "<div class='section'>";
    echo "<h2>5. Recommendations</h2>";
    echo "<ul>";
    echo "<li>✅ Ensure all hospital admin users have correct hospital_id assigned</li>";
    echo "<li>✅ All data (patients, doctors, appointments, etc.) should have hospital_id</li>";
    echo "<li>✅ Each hospital admin should only see their hospital's data</li>";
    echo "<li>✅ Super admin can see all hospitals' data</li>";
    echo "</ul>";
    echo "</div>";

} catch (PDOException $e) {
    echo "<p class='error'>Database Error: " . htmlspecialchars($e->getMessage()) . "</p>";
} catch (Exception $e) {
    echo "<p class='error'>Error: " . htmlspecialchars($e->getMessage()) . "</p>";
}

echo "</div>";
echo "</body></html>";

